/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.tieredstore.provider;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Stopwatch;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.rocketmq.logging.org.slf4j.Logger;
import org.apache.rocketmq.logging.org.slf4j.LoggerFactory;
import org.apache.rocketmq.tieredstore.common.AppendResult;
import org.apache.rocketmq.tieredstore.common.FileSegmentType;
import org.apache.rocketmq.tieredstore.common.TieredMessageStoreConfig;
import org.apache.rocketmq.tieredstore.exception.TieredStoreErrorCode;
import org.apache.rocketmq.tieredstore.exception.TieredStoreException;
import org.apache.rocketmq.tieredstore.provider.TieredStoreProvider;
import org.apache.rocketmq.tieredstore.provider.inputstream.TieredFileSegmentInputStream;
import org.apache.rocketmq.tieredstore.provider.inputstream.TieredFileSegmentInputStreamFactory;
import org.apache.rocketmq.tieredstore.util.MessageBufferUtil;

public abstract class TieredFileSegment
implements Comparable<TieredFileSegment>,
TieredStoreProvider {
    private static final Logger logger = LoggerFactory.getLogger((String)"RocketmqTieredStore");
    protected final String filePath;
    protected final long baseOffset;
    protected final FileSegmentType fileType;
    protected final TieredMessageStoreConfig storeConfig;
    private final long maxSize;
    private final ReentrantLock bufferLock;
    private final Semaphore commitLock;
    private volatile boolean full;
    private volatile boolean closed;
    private volatile long minTimestamp;
    private volatile long maxTimestamp;
    private volatile long commitPosition;
    private volatile long appendPosition;
    private volatile long dispatchCommitOffset = 0L;
    private ByteBuffer codaBuffer;
    private List<ByteBuffer> uploadBufferList = new ArrayList<ByteBuffer>();
    private CompletableFuture<Boolean> flightCommitRequest = CompletableFuture.completedFuture(false);

    public TieredFileSegment(TieredMessageStoreConfig storeConfig, FileSegmentType fileType, String filePath, long baseOffset) {
        this.storeConfig = storeConfig;
        this.fileType = fileType;
        this.filePath = filePath;
        this.baseOffset = baseOffset;
        this.closed = false;
        this.bufferLock = new ReentrantLock();
        this.commitLock = new Semaphore(1);
        this.commitPosition = 0L;
        this.appendPosition = 0L;
        this.minTimestamp = Long.MAX_VALUE;
        this.maxTimestamp = Long.MAX_VALUE;
        this.maxSize = this.getMaxSizeAccordingFileType(storeConfig);
    }

    private long getMaxSizeAccordingFileType(TieredMessageStoreConfig storeConfig) {
        switch (this.fileType) {
            case COMMIT_LOG: {
                return storeConfig.getTieredStoreCommitLogMaxSize();
            }
            case CONSUME_QUEUE: {
                return storeConfig.getTieredStoreConsumeQueueMaxSize();
            }
            case INDEX: {
                return Long.MAX_VALUE;
            }
        }
        throw new IllegalArgumentException("Unsupported file type: " + (Object)((Object)this.fileType));
    }

    @Override
    public int compareTo(TieredFileSegment o) {
        return Long.compare(this.baseOffset, o.baseOffset);
    }

    public long getBaseOffset() {
        return this.baseOffset;
    }

    public long getCommitOffset() {
        return this.baseOffset + this.commitPosition;
    }

    public long getCommitPosition() {
        return this.commitPosition;
    }

    public long getDispatchCommitOffset() {
        return this.dispatchCommitOffset;
    }

    public long getMaxOffset() {
        return this.baseOffset + this.appendPosition;
    }

    public long getMaxSize() {
        return this.maxSize;
    }

    public long getMinTimestamp() {
        return this.minTimestamp;
    }

    public void setMinTimestamp(long minTimestamp) {
        this.minTimestamp = minTimestamp;
    }

    public long getMaxTimestamp() {
        return this.maxTimestamp;
    }

    public void setMaxTimestamp(long maxTimestamp) {
        this.maxTimestamp = maxTimestamp;
    }

    public boolean isFull() {
        return this.full;
    }

    public void setFull() {
        this.setFull(true);
    }

    public void setFull(boolean appendCoda) {
        this.bufferLock.lock();
        try {
            this.full = true;
            if (this.fileType == FileSegmentType.COMMIT_LOG && appendCoda) {
                this.appendCoda();
            }
        }
        finally {
            this.bufferLock.unlock();
        }
    }

    public boolean isClosed() {
        return this.closed;
    }

    public void close() {
        this.closed = true;
    }

    public FileSegmentType getFileType() {
        return this.fileType;
    }

    public void initPosition(long pos) {
        this.commitPosition = pos;
        this.appendPosition = pos;
    }

    private List<ByteBuffer> rollingUploadBuffer() {
        this.bufferLock.lock();
        try {
            List<ByteBuffer> tmp = this.uploadBufferList;
            this.uploadBufferList = new ArrayList<ByteBuffer>();
            List<ByteBuffer> list = tmp;
            return list;
        }
        finally {
            this.bufferLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void sendBackBuffer(TieredFileSegmentInputStream inputStream) {
        this.bufferLock.lock();
        try {
            List<ByteBuffer> tmpBufferList = inputStream.getUploadBufferList();
            for (ByteBuffer buffer : tmpBufferList) {
                buffer.rewind();
            }
            tmpBufferList.addAll(this.uploadBufferList);
            this.uploadBufferList = tmpBufferList;
            if (inputStream.getCodaBuffer() != null) {
                this.codaBuffer.rewind();
            }
        }
        finally {
            this.bufferLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public AppendResult append(ByteBuffer byteBuf, long timeStamp) {
        if (this.closed) {
            return AppendResult.FILE_CLOSED;
        }
        this.bufferLock.lock();
        try {
            if (this.full || this.codaBuffer != null) {
                AppendResult appendResult = AppendResult.FILE_FULL;
                return appendResult;
            }
            if (this.fileType == FileSegmentType.INDEX) {
                this.minTimestamp = byteBuf.getLong(4);
                this.maxTimestamp = byteBuf.getLong(12);
                this.appendPosition += (long)byteBuf.remaining();
                this.uploadBufferList.add(byteBuf);
                this.setFull();
                AppendResult appendResult = AppendResult.SUCCESS;
                return appendResult;
            }
            if (this.appendPosition + (long)byteBuf.remaining() > this.maxSize) {
                this.setFull();
                AppendResult appendResult = AppendResult.FILE_FULL;
                return appendResult;
            }
            if (this.uploadBufferList.size() > this.storeConfig.getTieredStoreGroupCommitCount() || this.appendPosition - this.commitPosition > (long)this.storeConfig.getTieredStoreGroupCommitSize()) {
                this.commitAsync();
            }
            if (this.uploadBufferList.size() > this.storeConfig.getTieredStoreMaxGroupCommitCount()) {
                logger.debug("TieredFileSegment#append: buffer full: file: {}, upload buffer size: {}", (Object)this.getPath(), (Object)this.uploadBufferList.size());
                AppendResult appendResult = AppendResult.BUFFER_FULL;
                return appendResult;
            }
            if (timeStamp != Long.MAX_VALUE) {
                this.maxTimestamp = timeStamp;
                if (this.minTimestamp == Long.MAX_VALUE) {
                    this.minTimestamp = timeStamp;
                }
            }
            this.appendPosition += (long)byteBuf.remaining();
            this.uploadBufferList.add(byteBuf);
            AppendResult appendResult = AppendResult.SUCCESS;
            return appendResult;
        }
        finally {
            this.bufferLock.unlock();
        }
    }

    public void setCommitPosition(long commitPosition) {
        this.commitPosition = commitPosition;
    }

    public long getAppendPosition() {
        return this.appendPosition;
    }

    @VisibleForTesting
    public void setAppendPosition(long appendPosition) {
        this.appendPosition = appendPosition;
    }

    private void appendCoda() {
        if (this.codaBuffer != null) {
            return;
        }
        this.codaBuffer = ByteBuffer.allocate(16);
        this.codaBuffer.putInt(16);
        this.codaBuffer.putInt(-875286124);
        this.codaBuffer.putLong(this.maxTimestamp);
        this.codaBuffer.flip();
        this.appendPosition += 16L;
    }

    public ByteBuffer read(long position, int length) {
        return this.readAsync(position, length).join();
    }

    public CompletableFuture<ByteBuffer> readAsync(long position, int length) {
        CompletableFuture<ByteBuffer> future = new CompletableFuture<ByteBuffer>();
        if (position < 0L || length < 0) {
            future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.ILLEGAL_PARAM, "position or length is negative"));
            return future;
        }
        if (length == 0) {
            future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.ILLEGAL_PARAM, "length is zero"));
            return future;
        }
        if (position >= this.commitPosition) {
            future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.ILLEGAL_PARAM, "position is illegal"));
            return future;
        }
        if (position + (long)length > this.commitPosition) {
            logger.warn("TieredFileSegment#readAsync request position + length is greater than commit position, correct length using commit position, file: {}, request position: {}, commit position:{}, change length from {} to {}", new Object[]{this.getPath(), position, this.commitPosition, length, this.commitPosition - position});
            length = (int)(this.commitPosition - position);
            if (length == 0) {
                future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.NO_NEW_DATA, "request position is equal to commit position"));
                return future;
            }
            if (this.fileType == FileSegmentType.CONSUME_QUEUE && length % 20 != 0) {
                future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.ILLEGAL_PARAM, "position and length is illegal"));
                return future;
            }
        }
        return this.read0(position, length);
    }

    public boolean needCommit() {
        return this.appendPosition > this.commitPosition;
    }

    public boolean commit() {
        if (this.closed) {
            return false;
        }
        Boolean result = this.commitAsync().join();
        if (!result.booleanValue()) {
            result = this.flightCommitRequest.join();
        }
        return result;
    }

    public CompletableFuture<Boolean> commitAsync() {
        if (this.closed) {
            return CompletableFuture.completedFuture(false);
        }
        Stopwatch stopwatch = Stopwatch.createStarted();
        if (!this.needCommit()) {
            return CompletableFuture.completedFuture(true);
        }
        try {
            int permits = this.commitLock.drainPermits();
            if (permits <= 0) {
                return CompletableFuture.completedFuture(false);
            }
        }
        catch (Exception e2) {
            return CompletableFuture.completedFuture(false);
        }
        List<ByteBuffer> bufferList = this.rollingUploadBuffer();
        int bufferSize = 0;
        for (ByteBuffer buffer : bufferList) {
            bufferSize += buffer.remaining();
        }
        if (this.codaBuffer != null) {
            bufferSize += this.codaBuffer.remaining();
        }
        if (bufferSize == 0) {
            return CompletableFuture.completedFuture(true);
        }
        TieredFileSegmentInputStream inputStream = TieredFileSegmentInputStreamFactory.build(this.fileType, this.baseOffset + this.commitPosition, bufferList, this.codaBuffer, bufferSize);
        int finalBufferSize = bufferSize;
        try {
            this.flightCommitRequest = ((CompletableFuture)((CompletableFuture)this.commit0(inputStream, this.commitPosition, bufferSize, this.fileType != FileSegmentType.INDEX).thenApply(result -> {
                if (result.booleanValue()) {
                    if (this.fileType == FileSegmentType.COMMIT_LOG && bufferList.size() > 0) {
                        this.dispatchCommitOffset = MessageBufferUtil.getQueueOffset((ByteBuffer)bufferList.get(bufferList.size() - 1));
                    }
                    this.commitPosition += (long)finalBufferSize;
                    return true;
                }
                this.sendBackBuffer(inputStream);
                return false;
            })).exceptionally(e -> this.handleCommitException(inputStream, (Throwable)e))).whenComplete((result, e) -> {
                if (this.commitLock.availablePermits() == 0) {
                    logger.debug("TieredFileSegment#commitAsync: commit cost: {}ms, file: {}, item count: {}, buffer size: {}", new Object[]{stopwatch.elapsed(TimeUnit.MILLISECONDS), this.getPath(), bufferList.size(), finalBufferSize});
                    this.commitLock.release();
                } else {
                    logger.error("[Bug]TieredFileSegment#commitAsync: commit lock is already released: available permits: {}", (Object)this.commitLock.availablePermits());
                }
            });
            return this.flightCommitRequest;
        }
        catch (Exception e3) {
            this.handleCommitException(inputStream, e3);
            if (this.commitLock.availablePermits() == 0) {
                logger.debug("TieredFileSegment#commitAsync: commit cost: {}ms, file: {}, item count: {}, buffer size: {}", new Object[]{stopwatch.elapsed(TimeUnit.MILLISECONDS), this.getPath(), bufferList.size(), finalBufferSize});
                this.commitLock.release();
            } else {
                logger.error("[Bug]TieredFileSegment#commitAsync: commit lock is already released: available permits: {}", (Object)this.commitLock.availablePermits());
            }
            return CompletableFuture.completedFuture(false);
        }
    }

    private boolean handleCommitException(TieredFileSegmentInputStream inputStream, Throwable e) {
        Throwable cause = e.getCause() != null ? e.getCause() : e;
        this.sendBackBuffer(inputStream);
        long realSize = 0L;
        if (cause instanceof TieredStoreException && ((TieredStoreException)cause).getPosition() > 0L) {
            realSize = ((TieredStoreException)cause).getPosition();
        }
        if (realSize <= 0L) {
            realSize = this.getSize();
        }
        if (realSize > 0L && realSize > this.commitPosition) {
            logger.error("TieredFileSegment#handleCommitException: commit failed: file: {}, try to fix position: origin: {}, real: {}", new Object[]{this.getPath(), this.commitPosition, realSize, cause});
            long diff = this.appendPosition - this.commitPosition;
            this.commitPosition = realSize;
            this.appendPosition = realSize + diff;
        } else if (realSize < this.commitPosition) {
            logger.error("[Bug]TieredFileSegment#handleCommitException: commit failed: file: {}, can not fix position: origin: {}, real: {}", new Object[]{this.getPath(), this.commitPosition, realSize, cause});
        }
        return false;
    }
}

