/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.ozoneimpl;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.time.Instant;
import java.util.Iterator;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.hdfs.util.Canceler;
import org.apache.hadoop.hdfs.util.DataTransferThrottler;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.interfaces.Container;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.ozoneimpl.ContainerController;
import org.apache.hadoop.ozone.container.ozoneimpl.ContainerDataScrubberMetrics;
import org.apache.hadoop.ozone.container.ozoneimpl.ContainerScrubberConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerDataScanner
extends Thread {
    public static final Logger LOG = LoggerFactory.getLogger(ContainerDataScanner.class);
    private final HddsVolume volume;
    private final ContainerController controller;
    private final DataTransferThrottler throttler;
    private final Canceler canceler;
    private final ContainerDataScrubberMetrics metrics;
    private final long dataScanInterval;
    private static final String NAME_FORMAT = "ContainerDataScanner(%s)";
    private volatile boolean stopping = false;

    public ContainerDataScanner(ContainerScrubberConfiguration conf, ContainerController controller, HddsVolume volume) {
        this.controller = controller;
        this.volume = volume;
        this.dataScanInterval = conf.getDataScanInterval();
        this.throttler = new HddsDataTransferThrottler(conf.getBandwidthPerVolume());
        this.canceler = new Canceler();
        this.metrics = ContainerDataScrubberMetrics.create(volume.toString());
        this.setName(String.format(NAME_FORMAT, volume));
        this.setDaemon(true);
    }

    @Override
    public void run() {
        if (LOG.isTraceEnabled()) {
            LOG.trace("{}: thread starting.", (Object)this);
        }
        try {
            while (!this.stopping) {
                this.runIteration();
                this.metrics.resetNumContainersScanned();
                this.metrics.resetNumUnhealthyContainers();
            }
            LOG.info("{} exiting.", (Object)this);
        }
        catch (Exception e) {
            LOG.error("{} exiting because of exception ", (Object)this, (Object)e);
        }
        finally {
            if (this.metrics != null) {
                this.metrics.unregister();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public void runIteration() {
        long startTime = System.nanoTime();
        Iterator<Container<?>> itr = this.controller.getContainers(this.volume);
        while (!this.stopping && itr.hasNext()) {
            Container<?> c = itr.next();
            if (!c.shouldScanData()) continue;
            Object containerData = c.getContainerData();
            long containerId = ((ContainerData)containerData).getContainerID();
            try {
                ContainerDataScanner.logScanStart(containerData);
                if (!c.scanData(this.throttler, this.canceler)) {
                    this.metrics.incNumUnHealthyContainers();
                    this.controller.markContainerUnhealthy(containerId);
                    continue;
                }
                Instant now = Instant.now();
                ContainerDataScanner.logScanCompleted(containerData, now);
                this.controller.updateDataScanTimestamp(containerId, now);
            }
            catch (IOException ex) {
                LOG.warn("Unexpected exception while scanning container " + containerId, (Throwable)ex);
            }
            finally {
                this.metrics.incNumContainersScanned();
            }
        }
        long totalDuration = System.nanoTime() - startTime;
        if (!this.stopping) {
            long elapsedMillis;
            long remainingSleep;
            if (this.metrics.getNumContainersScanned() > 0) {
                this.metrics.incNumScanIterations();
                LOG.info("Completed an iteration of container data scrubber in {} minutes. Number of iterations (since the data-node restart) : {}, Number of containers scanned in this iteration : {}, Number of unhealthy containers found in this iteration : {}", new Object[]{TimeUnit.NANOSECONDS.toMinutes(totalDuration), this.metrics.getNumScanIterations(), this.metrics.getNumContainersScanned(), this.metrics.getNumUnHealthyContainers()});
            }
            if ((remainingSleep = this.dataScanInterval - (elapsedMillis = TimeUnit.NANOSECONDS.toMillis(totalDuration))) > 0L) {
                try {
                    Thread.sleep(remainingSleep);
                }
                catch (InterruptedException ignored) {
                    LOG.warn("Operation was interrupted.");
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

    private static void logScanStart(ContainerData containerData) {
        if (LOG.isDebugEnabled()) {
            Optional<Instant> scanTimestamp = containerData.lastDataScanTime();
            String lastScanTime = scanTimestamp.map(ts -> "at " + ts).orElse("never");
            LOG.debug("Scanning container {}, last scanned {}", (Object)containerData.getContainerID(), (Object)lastScanTime);
        }
    }

    private static void logScanCompleted(ContainerData containerData, Instant timestamp) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Completed scan of container {} at {}", (Object)containerData.getContainerID(), (Object)timestamp);
        }
    }

    public synchronized void shutdown() {
        this.stopping = true;
        this.canceler.cancel(String.format(NAME_FORMAT, this.volume) + " is shutting down");
        this.interrupt();
        try {
            this.join();
        }
        catch (InterruptedException ex) {
            LOG.warn("Unexpected exception while stopping data scanner for volume " + this.volume, (Throwable)ex);
            Thread.currentThread().interrupt();
        }
    }

    @VisibleForTesting
    public ContainerDataScrubberMetrics getMetrics() {
        return this.metrics;
    }

    @Override
    public String toString() {
        return String.format(NAME_FORMAT, this.volume + ", " + this.volume.getStorageID());
    }

    private class HddsDataTransferThrottler
    extends DataTransferThrottler {
        HddsDataTransferThrottler(long bandwidthPerSec) {
            super(bandwidthPerSec);
        }

        public synchronized void throttle(long numOfBytes) {
            ContainerDataScanner.this.metrics.incNumBytesScanned(numOfBytes);
            super.throttle(numOfBytes);
        }

        public synchronized void throttle(long numOfBytes, Canceler c) {
            ContainerDataScanner.this.metrics.incNumBytesScanned(numOfBytes);
            super.throttle(numOfBytes, c);
        }
    }
}

