/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue.impl;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.StorageUnit;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.utils.MetadataKeyFilters;
import org.apache.hadoop.hdds.utils.db.BatchOperation;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.ozone.container.common.helpers.BlockData;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.interfaces.Container;
import org.apache.hadoop.ozone.container.common.utils.ContainerCache;
import org.apache.hadoop.ozone.container.common.utils.ReferenceCountedDB;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainer;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.BlockUtils;
import org.apache.hadoop.ozone.container.keyvalue.interfaces.BlockManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlockManagerImpl
implements BlockManager {
    static final Logger LOG = LoggerFactory.getLogger(BlockManagerImpl.class);
    private ConfigurationSource config;
    private static final String DB_NULL_ERR_MSG = "DB cannot be null here";
    private static final String NO_SUCH_BLOCK_ERR_MSG = "Unable to find the block.";
    private final long defaultReadBufferCapacity;

    public BlockManagerImpl(ConfigurationSource conf) {
        Preconditions.checkNotNull((Object)conf, (Object)"Config cannot be null");
        this.config = conf;
        this.defaultReadBufferCapacity = (long)this.config.getStorageSize("ozone.chunk.read.buffer.default.size", "64KB", StorageUnit.BYTES);
    }

    @Override
    public long putBlock(Container container, BlockData data) throws IOException {
        return this.putBlock(container, data, true);
    }

    @Override
    public long putBlock(Container container, BlockData data, boolean incrKeyCount) throws IOException {
        return BlockManagerImpl.persistPutBlock((KeyValueContainer)container, data, this.config, incrKeyCount);
    }

    public static long persistPutBlock(KeyValueContainer container, BlockData data, ConfigurationSource config, boolean incrKeyCount) throws IOException {
        Preconditions.checkNotNull((Object)data, (Object)"BlockData cannot be null for put operation.");
        Preconditions.checkState((data.getContainerID() >= 0L ? 1 : 0) != 0, (Object)"Container Id cannot be negative");
        try (ReferenceCountedDB db = BlockUtils.getDB(container.getContainerData(), config);){
            Preconditions.checkNotNull((Object)db, (Object)DB_NULL_ERR_MSG);
            long bcsId = data.getBlockCommitSequenceId();
            long containerBCSId = container.getContainerData().getBlockCommitSequenceId();
            if (bcsId != 0L && bcsId <= containerBCSId) {
                LOG.debug("blockCommitSequenceId {} in the Container Db is greater than the supplied value {}. Ignoring it", (Object)containerBCSId, (Object)bcsId);
                long l = data.getSize();
                return l;
            }
            try (BatchOperation batch = db.getStore().getBatchHandler().initBatchOperation();){
                db.getStore().getBlockDataTable().putWithBatch(batch, (Object)Long.toString(data.getLocalID()), (Object)data);
                if (bcsId != 0L) {
                    db.getStore().getMetadataTable().putWithBatch(batch, (Object)"#BCSID", (Object)bcsId);
                }
                db.getStore().getMetadataTable().putWithBatch(batch, (Object)"#BYTESUSED", (Object)container.getContainerData().getBytesUsed());
                if (incrKeyCount) {
                    db.getStore().getMetadataTable().putWithBatch(batch, (Object)"#BLOCKCOUNT", (Object)(container.getContainerData().getKeyCount() + 1L));
                }
                db.getStore().getBatchHandler().commitBatchOperation(batch);
            }
            if (bcsId != 0L) {
                container.updateBlockCommitSequenceId(bcsId);
            }
            if (incrKeyCount) {
                container.getContainerData().incrKeyCount();
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("Block " + data.getBlockID() + " successfully committed with bcsId " + bcsId + " chunk size " + data.getChunks().size());
            }
            long l = data.getSize();
            return l;
        }
    }

    @Override
    public BlockData getBlock(Container container, BlockID blockID) throws IOException {
        long bcsId = blockID.getBlockCommitSequenceId();
        Preconditions.checkNotNull((Object)blockID, (Object)"BlockID cannot be null in GetBlock request");
        Preconditions.checkNotNull((Object)container, (Object)"Container cannot be null");
        KeyValueContainerData containerData = (KeyValueContainerData)container.getContainerData();
        try (ReferenceCountedDB db = BlockUtils.getDB(containerData, this.config);){
            Preconditions.checkNotNull((Object)db, (Object)DB_NULL_ERR_MSG);
            long containerBCSId = containerData.getBlockCommitSequenceId();
            if (containerBCSId < bcsId) {
                throw new StorageContainerException("Unable to find the block with bcsID " + bcsId + " .Container " + ((ContainerData)container.getContainerData()).getContainerID() + " bcsId is " + containerBCSId + ".", ContainerProtos.Result.UNKNOWN_BCSID);
            }
            BlockData blockData = this.getBlockByID(db, blockID);
            long id = blockData.getBlockID().getBlockCommitSequenceId();
            if (id < bcsId) {
                throw new StorageContainerException("bcsId " + bcsId + " mismatches with existing block Id " + id + " for block " + blockID + ".", ContainerProtos.Result.BCSID_MISMATCH);
            }
            BlockData blockData2 = blockData;
            return blockData2;
        }
    }

    @Override
    public long getCommittedBlockLength(Container container, BlockID blockID) throws IOException {
        KeyValueContainerData containerData = (KeyValueContainerData)container.getContainerData();
        try (ReferenceCountedDB db = BlockUtils.getDB(containerData, this.config);){
            Preconditions.checkNotNull((Object)db, (Object)DB_NULL_ERR_MSG);
            BlockData blockData = this.getBlockByID(db, blockID);
            long l = blockData.getSize();
            return l;
        }
    }

    @Override
    public long getDefaultReadBufferCapacity() {
        return this.defaultReadBufferCapacity;
    }

    @Override
    public void deleteBlock(Container container, BlockID blockID) throws IOException {
        Preconditions.checkNotNull((Object)blockID, (Object)"block ID cannot be null.");
        Preconditions.checkState((blockID.getContainerID() >= 0L ? 1 : 0) != 0, (Object)"Container ID cannot be negative.");
        Preconditions.checkState((blockID.getLocalID() >= 0L ? 1 : 0) != 0, (Object)"Local ID cannot be negative.");
        KeyValueContainerData cData = (KeyValueContainerData)container.getContainerData();
        try (ReferenceCountedDB db = BlockUtils.getDB(cData, this.config);){
            Preconditions.checkNotNull((Object)db, (Object)DB_NULL_ERR_MSG);
            this.getBlockByID(db, blockID);
            try (BatchOperation batch = db.getStore().getBatchHandler().initBatchOperation();){
                String localID = Long.toString(blockID.getLocalID());
                db.getStore().getBlockDataTable().deleteWithBatch(batch, (Object)localID);
                long blockCount = ((ContainerData)container.getContainerData()).getKeyCount() - 1L;
                db.getStore().getMetadataTable().putWithBatch(batch, (Object)"#BLOCKCOUNT", (Object)blockCount);
                db.getStore().getBatchHandler().commitBatchOperation(batch);
            }
            ((ContainerData)container.getContainerData()).decrKeyCount();
        }
    }

    /*
     * Loose catch block
     */
    @Override
    public List<BlockData> listBlock(Container container, long startLocalID, int count) throws IOException {
        Preconditions.checkNotNull((Object)container, (Object)"container cannot be null");
        Preconditions.checkState((startLocalID >= 0L ? 1 : 0) != 0, (Object)"startLocal ID cannot be negative");
        Preconditions.checkArgument((count > 0 ? 1 : 0) != 0, (Object)"Count must be a positive number.");
        container.readLock();
        try {
            ArrayList<BlockData> result = null;
            KeyValueContainerData cData = (KeyValueContainerData)container.getContainerData();
            try (ReferenceCountedDB db = BlockUtils.getDB(cData, this.config);){
                result = new ArrayList<BlockData>();
                List range = db.getStore().getBlockDataTable().getSequentialRangeKVs((Object)Long.toString(startLocalID), count, new MetadataKeyFilters.MetadataKeyFilter[]{MetadataKeyFilters.getUnprefixedKeyFilter()});
                for (Table.KeyValue entry : range) {
                    BlockData data = new BlockData(((BlockData)entry.getValue()).getBlockID());
                    result.add(data);
                }
                ArrayList<BlockData> arrayList = result;
                return arrayList;
            }
            {
                catch (Throwable throwable) {
                    throw throwable;
                }
            }
        }
        finally {
            container.readUnlock();
        }
    }

    @Override
    public void shutdown() {
        BlockUtils.shutdownCache(ContainerCache.getInstance(this.config));
    }

    private BlockData getBlockByID(ReferenceCountedDB db, BlockID blockID) throws IOException {
        String blockKey = Long.toString(blockID.getLocalID());
        BlockData blockData = (BlockData)db.getStore().getBlockDataTable().get((Object)blockKey);
        if (blockData == null) {
            throw new StorageContainerException(NO_SUCH_BLOCK_ERR_MSG, ContainerProtos.Result.NO_SUCH_BLOCK);
        }
        return blockData;
    }
}

