/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.volume;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.util.Objects;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nullable;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdds.annotation.InterfaceAudience;
import org.apache.hadoop.hdds.annotation.InterfaceStability;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.fs.SpaceUsageCheckFactory;
import org.apache.hadoop.hdfs.server.datanode.StorageLocation;
import org.apache.hadoop.hdfs.server.datanode.checker.Checkable;
import org.apache.hadoop.hdfs.server.datanode.checker.VolumeCheckResult;
import org.apache.hadoop.ozone.common.InconsistentStorageStateException;
import org.apache.hadoop.ozone.container.common.DataNodeLayoutVersion;
import org.apache.hadoop.ozone.container.common.helpers.DatanodeVersionFile;
import org.apache.hadoop.ozone.container.common.utils.HddsVolumeUtil;
import org.apache.hadoop.ozone.container.common.volume.VolumeIOStats;
import org.apache.hadoop.ozone.container.common.volume.VolumeInfo;
import org.apache.hadoop.util.DiskChecker;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class HddsVolume
implements Checkable<Boolean, VolumeCheckResult> {
    private static final Logger LOG = LoggerFactory.getLogger(HddsVolume.class);
    public static final String HDDS_VOLUME_DIR = "hdds";
    private final File hddsRootDir;
    private final VolumeInfo volumeInfo;
    private VolumeState state;
    private final VolumeIOStats volumeIOStats;
    private String storageID;
    private String clusterID;
    private String datanodeUuid;
    private long cTime;
    private int layoutVersion;
    private final AtomicLong committedBytes;

    public VolumeCheckResult check(@Nullable Boolean unused) throws Exception {
        if (!this.hddsRootDir.exists()) {
            return VolumeCheckResult.FAILED;
        }
        DiskChecker.checkDir((File)this.hddsRootDir);
        return VolumeCheckResult.HEALTHY;
    }

    private HddsVolume(Builder b) throws IOException {
        if (!b.failedVolume) {
            StorageLocation location = StorageLocation.parse((String)b.volumeRootStr);
            this.hddsRootDir = new File(location.getUri().getPath(), HDDS_VOLUME_DIR);
            this.state = VolumeState.NOT_INITIALIZED;
            this.clusterID = b.clusterID;
            this.datanodeUuid = b.datanodeUuid;
            this.volumeIOStats = new VolumeIOStats(b.volumeRootStr);
            this.volumeInfo = new VolumeInfo.Builder(b.volumeRootStr, b.conf).storageType(b.storageType).usageCheckFactory(b.usageCheckFactory).build();
            this.committedBytes = new AtomicLong(0L);
            LOG.info("Creating Volume: {} of storage type : {} and capacity : {}", new Object[]{this.hddsRootDir, b.storageType, this.volumeInfo.getCapacity()});
            this.initialize();
        } else {
            this.hddsRootDir = new File(b.volumeRootStr);
            this.volumeIOStats = null;
            this.volumeInfo = null;
            this.storageID = UUID.randomUUID().toString();
            this.state = VolumeState.FAILED;
            this.committedBytes = null;
        }
    }

    public VolumeInfo getVolumeInfo() {
        return this.volumeInfo;
    }

    private void initialize() throws IOException {
        VolumeState intialVolumeState = this.analyzeVolumeState();
        switch (intialVolumeState) {
            case NON_EXISTENT: {
                if (!this.hddsRootDir.mkdirs()) {
                    throw new IOException("Cannot create directory " + this.hddsRootDir);
                }
                this.setState(VolumeState.NOT_FORMATTED);
                this.createVersionFile();
                break;
            }
            case NOT_FORMATTED: {
                this.createVersionFile();
                break;
            }
            case NOT_INITIALIZED: {
                this.readVersionFile();
                this.setState(VolumeState.NORMAL);
                break;
            }
            case INCONSISTENT: {
                throw new IOException("Volume is in an " + (Object)((Object)VolumeState.INCONSISTENT) + " state. Skipped loading volume: " + this.hddsRootDir.getPath());
            }
            default: {
                throw new IOException("Unrecognized initial state : " + (Object)((Object)intialVolumeState) + "of volume : " + this.hddsRootDir);
            }
        }
    }

    private VolumeState analyzeVolumeState() {
        if (!this.hddsRootDir.exists()) {
            return VolumeState.NON_EXISTENT;
        }
        if (!this.hddsRootDir.isDirectory()) {
            LOG.warn("Volume {} exists but is not a directory, current volume state: {}.", (Object)this.hddsRootDir.getPath(), (Object)VolumeState.INCONSISTENT);
            return VolumeState.INCONSISTENT;
        }
        File[] files = this.hddsRootDir.listFiles();
        if (files == null || files.length == 0) {
            return VolumeState.NOT_FORMATTED;
        }
        if (!this.getVersionFile().exists()) {
            LOG.warn("VERSION file does not exist in volume {}, current volume state: {}.", (Object)this.hddsRootDir.getPath(), (Object)VolumeState.INCONSISTENT);
            return VolumeState.INCONSISTENT;
        }
        return VolumeState.NOT_INITIALIZED;
    }

    public void format(String cid) throws IOException {
        Preconditions.checkNotNull((Object)cid, (Object)"clusterID cannot be null while formatting Volume");
        this.clusterID = cid;
        this.initialize();
    }

    private void createVersionFile() throws IOException {
        this.storageID = HddsVolumeUtil.generateUuid();
        this.cTime = Time.now();
        this.layoutVersion = DataNodeLayoutVersion.getLatestVersion().getVersion();
        if (this.clusterID == null || this.datanodeUuid == null) {
            LOG.debug("ClusterID not available. Cannot format the volume {}", (Object)this.hddsRootDir.getPath());
            this.setState(VolumeState.NOT_FORMATTED);
        } else {
            this.writeVersionFile();
            this.setState(VolumeState.NORMAL);
        }
    }

    private void writeVersionFile() throws IOException {
        Preconditions.checkNotNull((Object)this.storageID, (Object)"StorageID cannot be null in Version File");
        Preconditions.checkNotNull((Object)this.clusterID, (Object)"ClusterID cannot be null in Version File");
        Preconditions.checkNotNull((Object)this.datanodeUuid, (Object)"DatanodeUUID cannot be null in Version File");
        Preconditions.checkArgument((this.cTime > 0L ? 1 : 0) != 0, (Object)"Creation Time should be positive");
        Preconditions.checkArgument((this.layoutVersion == DataNodeLayoutVersion.getLatestVersion().getVersion() ? 1 : 0) != 0, (Object)"Version File should have the latest LayOutVersion");
        File versionFile = this.getVersionFile();
        LOG.debug("Writing Version file to disk, {}", (Object)versionFile);
        DatanodeVersionFile dnVersionFile = new DatanodeVersionFile(this.storageID, this.clusterID, this.datanodeUuid, this.cTime, this.layoutVersion);
        dnVersionFile.createVersionFile(versionFile);
    }

    private void readVersionFile() throws IOException {
        File versionFile = this.getVersionFile();
        Properties props = DatanodeVersionFile.readFrom(versionFile);
        if (props.isEmpty()) {
            throw new InconsistentStorageStateException("Version file " + versionFile + " is missing");
        }
        LOG.debug("Reading Version file from disk, {}", (Object)versionFile);
        this.storageID = HddsVolumeUtil.getStorageID(props, versionFile);
        this.clusterID = HddsVolumeUtil.getClusterID(props, versionFile, this.clusterID);
        this.datanodeUuid = HddsVolumeUtil.getDatanodeUUID(props, versionFile, this.datanodeUuid);
        this.cTime = HddsVolumeUtil.getCreationTime(props, versionFile);
        this.layoutVersion = HddsVolumeUtil.getLayOutVersion(props, versionFile);
    }

    private File getVersionFile() {
        return HddsVolumeUtil.getVersionFile(this.hddsRootDir);
    }

    public File getHddsRootDir() {
        return this.hddsRootDir;
    }

    public StorageType getStorageType() {
        if (this.volumeInfo != null) {
            return this.volumeInfo.getStorageType();
        }
        return StorageType.DEFAULT;
    }

    public String getStorageID() {
        return this.storageID;
    }

    public String getClusterID() {
        return this.clusterID;
    }

    public String getDatanodeUuid() {
        return this.datanodeUuid;
    }

    public long getCTime() {
        return this.cTime;
    }

    public int getLayoutVersion() {
        return this.layoutVersion;
    }

    public VolumeState getStorageState() {
        return this.state;
    }

    public long getCapacity() {
        return this.volumeInfo != null ? this.volumeInfo.getCapacity() : 0L;
    }

    public long getAvailable() {
        return this.volumeInfo != null ? this.volumeInfo.getAvailable() : 0L;
    }

    public long getUsedSpace() {
        return this.volumeInfo != null ? this.volumeInfo.getScmUsed() : 0L;
    }

    public void setState(VolumeState state) {
        this.state = state;
    }

    public boolean isFailed() {
        return this.state == VolumeState.FAILED;
    }

    public VolumeIOStats getVolumeIOStats() {
        return this.volumeIOStats;
    }

    public void failVolume() {
        this.setState(VolumeState.FAILED);
        if (this.volumeInfo != null) {
            this.volumeInfo.shutdownUsageThread();
        }
        if (this.volumeIOStats != null) {
            this.volumeIOStats.unregister();
        }
    }

    public void shutdown() {
        this.state = VolumeState.NON_EXISTENT;
        if (this.volumeInfo != null) {
            this.volumeInfo.shutdownUsageThread();
        }
        if (this.volumeIOStats != null) {
            this.volumeIOStats.unregister();
        }
    }

    public long incCommittedBytes(long delta) {
        return this.committedBytes.addAndGet(delta);
    }

    public long getCommittedBytes() {
        return this.committedBytes.get();
    }

    public int hashCode() {
        return Objects.hash(this.hddsRootDir);
    }

    public boolean equals(Object other) {
        return this == other || other instanceof HddsVolume && ((HddsVolume)other).hddsRootDir.equals(this.hddsRootDir);
    }

    public String toString() {
        return this.getHddsRootDir().toString();
    }

    public static enum VolumeState {
        NORMAL,
        FAILED,
        NON_EXISTENT,
        INCONSISTENT,
        NOT_FORMATTED,
        NOT_INITIALIZED;

    }

    public static class Builder {
        private final String volumeRootStr;
        private ConfigurationSource conf;
        private StorageType storageType;
        private String datanodeUuid;
        private String clusterID;
        private boolean failedVolume = false;
        private SpaceUsageCheckFactory usageCheckFactory;

        public Builder(String rootDirStr) {
            this.volumeRootStr = rootDirStr;
        }

        public Builder conf(ConfigurationSource config) {
            this.conf = config;
            return this;
        }

        public Builder storageType(StorageType st) {
            this.storageType = st;
            return this;
        }

        public Builder datanodeUuid(String datanodeUUID) {
            this.datanodeUuid = datanodeUUID;
            return this;
        }

        public Builder clusterID(String cid) {
            this.clusterID = cid;
            return this;
        }

        public Builder failedVolume(boolean failed) {
            this.failedVolume = failed;
            return this;
        }

        public Builder usageCheckFactory(SpaceUsageCheckFactory factory) {
            this.usageCheckFactory = factory;
            return this;
        }

        public HddsVolume build() throws IOException {
            return new HddsVolume(this);
        }
    }
}

