/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.statemachine;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.Closeable;
import java.io.IOException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.hdds.HddsUtils;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.security.x509.certificate.client.CertificateClient;
import org.apache.hadoop.hdds.utils.LegacyHadoopConfigurationSource;
import org.apache.hadoop.ozone.HddsDatanodeStopService;
import org.apache.hadoop.ozone.container.common.report.ReportManager;
import org.apache.hadoop.ozone.container.common.statemachine.DatanodeConfiguration;
import org.apache.hadoop.ozone.container.common.statemachine.SCMConnectionManager;
import org.apache.hadoop.ozone.container.common.statemachine.StateContext;
import org.apache.hadoop.ozone.container.common.statemachine.commandhandler.CloseContainerCommandHandler;
import org.apache.hadoop.ozone.container.common.statemachine.commandhandler.ClosePipelineCommandHandler;
import org.apache.hadoop.ozone.container.common.statemachine.commandhandler.CommandDispatcher;
import org.apache.hadoop.ozone.container.common.statemachine.commandhandler.CreatePipelineCommandHandler;
import org.apache.hadoop.ozone.container.common.statemachine.commandhandler.DeleteBlocksCommandHandler;
import org.apache.hadoop.ozone.container.common.statemachine.commandhandler.DeleteContainerCommandHandler;
import org.apache.hadoop.ozone.container.common.statemachine.commandhandler.ReplicateContainerCommandHandler;
import org.apache.hadoop.ozone.container.common.statemachine.commandhandler.SetNodeOperationalStateCommandHandler;
import org.apache.hadoop.ozone.container.keyvalue.TarContainerPacker;
import org.apache.hadoop.ozone.container.ozoneimpl.OzoneContainer;
import org.apache.hadoop.ozone.container.replication.ContainerReplicator;
import org.apache.hadoop.ozone.container.replication.DownloadAndImportReplicator;
import org.apache.hadoop.ozone.container.replication.MeasuredReplicator;
import org.apache.hadoop.ozone.container.replication.ReplicationSupervisor;
import org.apache.hadoop.ozone.container.replication.SimpleContainerDownloader;
import org.apache.hadoop.ozone.protocol.commands.SCMCommand;
import org.apache.hadoop.util.JvmPauseMonitor;
import org.apache.hadoop.util.Time;
import org.apache.ratis.util.ExitUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatanodeStateMachine
implements Closeable {
    @VisibleForTesting
    static final Logger LOG = LoggerFactory.getLogger(DatanodeStateMachine.class);
    private final ExecutorService executorService;
    private final ConfigurationSource conf;
    private final SCMConnectionManager connectionManager;
    private StateContext context;
    private final OzoneContainer container;
    private DatanodeDetails datanodeDetails;
    private final CommandDispatcher commandDispatcher;
    private final ReportManager reportManager;
    private long commandsHandled;
    private AtomicLong nextHB;
    private Thread stateMachineThread = null;
    private Thread cmdProcessThread = null;
    private final ReplicationSupervisor supervisor;
    private JvmPauseMonitor jvmPauseMonitor;
    private CertificateClient dnCertClient;
    private final HddsDatanodeStopService hddsDatanodeStopService;
    private final ReadWriteLock constructionLock = new ReentrantReadWriteLock();
    private final MeasuredReplicator replicatorMetrics;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DatanodeStateMachine(DatanodeDetails datanodeDetails, ConfigurationSource conf, CertificateClient certClient, HddsDatanodeStopService hddsDatanodeStopService) throws IOException {
        DatanodeConfiguration dnConf = (DatanodeConfiguration)conf.getObject(DatanodeConfiguration.class);
        this.hddsDatanodeStopService = hddsDatanodeStopService;
        this.conf = conf;
        this.datanodeDetails = datanodeDetails;
        this.executorService = Executors.newFixedThreadPool(this.getEndPointTaskThreadPoolSize(), new ThreadFactoryBuilder().setNameFormat("Datanode State Machine Task Thread - %d").build());
        this.connectionManager = new SCMConnectionManager(conf);
        this.context = new StateContext(this.conf, DatanodeStates.getInitState(), this);
        this.constructionLock.writeLock().lock();
        try {
            this.container = new OzoneContainer(this.datanodeDetails, conf, this.context, certClient);
        }
        finally {
            this.constructionLock.writeLock().unlock();
        }
        this.dnCertClient = certClient;
        this.nextHB = new AtomicLong(Time.monotonicNow());
        DownloadAndImportReplicator replicator = new DownloadAndImportReplicator(this.container.getContainerSet(), this.container.getController(), new SimpleContainerDownloader(conf, this.dnCertClient), new TarContainerPacker());
        this.replicatorMetrics = new MeasuredReplicator(replicator);
        this.supervisor = new ReplicationSupervisor(this.container.getContainerSet(), (ContainerReplicator)this.replicatorMetrics, dnConf.getReplicationMaxStreams());
        this.commandDispatcher = CommandDispatcher.newBuilder().addHandler(new CloseContainerCommandHandler()).addHandler(new DeleteBlocksCommandHandler(this.container.getContainerSet(), conf)).addHandler(new ReplicateContainerCommandHandler(conf, this.supervisor)).addHandler(new DeleteContainerCommandHandler(dnConf.getContainerDeleteThreads())).addHandler(new ClosePipelineCommandHandler()).addHandler(new CreatePipelineCommandHandler(conf)).addHandler(new SetNodeOperationalStateCommandHandler(conf)).setConnectionManager(this.connectionManager).setContainer(this.container).setContext(this.context).build();
        this.reportManager = ReportManager.newBuilder(conf).setStateContext(this.context).addPublisherFor(StorageContainerDatanodeProtocolProtos.NodeReportProto.class).addPublisherFor(StorageContainerDatanodeProtocolProtos.ContainerReportsProto.class).addPublisherFor(StorageContainerDatanodeProtocolProtos.CommandStatusReportsProto.class).addPublisherFor(StorageContainerDatanodeProtocolProtos.PipelineReportsProto.class).build();
    }

    private int getEndPointTaskThreadPoolSize() {
        int reconServerCount;
        int totalServerCount = reconServerCount = 1;
        try {
            totalServerCount += HddsUtils.getSCMAddresses((ConfigurationSource)this.conf).size();
        }
        catch (Exception e) {
            LOG.error("Fail to get scm addresses", (Throwable)e);
        }
        return totalServerCount;
    }

    public DatanodeDetails getDatanodeDetails() {
        return this.datanodeDetails;
    }

    public SCMConnectionManager getConnectionManager() {
        return this.connectionManager;
    }

    public OzoneContainer getContainer() {
        this.constructionLock.readLock().lock();
        try {
            OzoneContainer ozoneContainer = this.container;
            return ozoneContainer;
        }
        finally {
            this.constructionLock.readLock().unlock();
        }
    }

    private void start() throws IOException {
        long now = 0L;
        this.reportManager.init();
        this.initCommandHandlerThread(this.conf);
        this.jvmPauseMonitor = new JvmPauseMonitor();
        this.jvmPauseMonitor.init(LegacyHadoopConfigurationSource.asHadoopConfiguration((ConfigurationSource)this.conf));
        this.jvmPauseMonitor.start();
        while (this.context.getState() != DatanodeStates.SHUTDOWN) {
            try {
                LOG.debug("Executing cycle Number : {}", (Object)this.context.getExecutionCount());
                long heartbeatFrequency = this.context.getHeartbeatFrequency();
                this.nextHB.set(Time.monotonicNow() + heartbeatFrequency);
                this.context.execute(this.executorService, heartbeatFrequency, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            catch (Exception e) {
                LOG.error("Unable to finish the execution.", (Throwable)e);
            }
            if ((now = Time.monotonicNow()) >= this.nextHB.get() || Thread.interrupted()) continue;
            try {
                Thread.sleep(this.nextHB.get() - now);
            }
            catch (InterruptedException interruptedException) {}
        }
        if (this.context.getShutdownOnError()) {
            LOG.error("DatanodeStateMachine Shutdown due to an critical error");
            this.hddsDatanodeStopService.stopService();
        }
    }

    public StateContext getContext() {
        return this.context;
    }

    public void setContext(StateContext context) {
        this.context = context;
    }

    @Override
    public void close() throws IOException {
        if (this.stateMachineThread != null) {
            this.stateMachineThread.interrupt();
        }
        if (this.cmdProcessThread != null) {
            this.cmdProcessThread.interrupt();
        }
        this.context.setState(DatanodeStates.getLastState());
        this.executorService.shutdown();
        try {
            if (!this.executorService.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.executorService.shutdownNow();
            }
            if (!this.executorService.awaitTermination(5L, TimeUnit.SECONDS)) {
                LOG.error("Unable to shutdown state machine properly.");
            }
        }
        catch (InterruptedException e) {
            LOG.error("Error attempting to shutdown.", (Throwable)e);
            this.executorService.shutdownNow();
            Thread.currentThread().interrupt();
        }
        if (this.connectionManager != null) {
            this.connectionManager.close();
        }
        if (this.container != null) {
            this.container.stop();
        }
        if (this.jvmPauseMonitor != null) {
            this.jvmPauseMonitor.stop();
        }
        if (this.commandDispatcher != null) {
            this.commandDispatcher.stop();
        }
    }

    public void startDaemon() {
        Runnable startStateMachineTask = () -> {
            try {
                this.start();
                LOG.info("Ozone container server started.");
            }
            catch (Exception ex) {
                LOG.error("Unable to start the DatanodeState Machine", (Throwable)ex);
            }
        };
        this.stateMachineThread = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("Datanode State Machine Thread - %d").setUncaughtExceptionHandler((t, ex) -> {
            String message = "Terminate Datanode, encounter uncaught exception in Datanode State Machine Thread";
            ExitUtils.terminate((int)1, (String)message, (Throwable)ex, (Logger)LOG);
        }).build().newThread(startStateMachineTask);
        this.stateMachineThread.start();
    }

    public void triggerHeartbeat() {
        if (this.stateMachineThread != null) {
            this.stateMachineThread.interrupt();
        }
    }

    public void join() throws InterruptedException {
        if (this.stateMachineThread != null) {
            this.stateMachineThread.join();
        }
        if (this.cmdProcessThread != null) {
            this.cmdProcessThread.join();
        }
    }

    public synchronized void stopDaemon() {
        try {
            try {
                this.replicatorMetrics.close();
            }
            catch (Exception e) {
                LOG.error("Couldn't stop replicator metrics", (Throwable)e);
            }
            this.supervisor.stop();
            this.context.setShutdownGracefully();
            this.context.setState(DatanodeStates.SHUTDOWN);
            this.reportManager.shutdown();
            this.close();
            LOG.info("Ozone container server stopped.");
        }
        catch (IOException e) {
            LOG.error("Stop ozone container server failed.", (Throwable)e);
        }
    }

    @VisibleForTesting
    public boolean isDaemonStopped() {
        return this.executorService.isShutdown() && this.getContext().getState() == DatanodeStates.SHUTDOWN;
    }

    private void initCommandHandlerThread(ConfigurationSource config) {
        Runnable processCommandQueue = () -> {
            while (this.getContext().getState() != DatanodeStates.SHUTDOWN) {
                SCMCommand command = this.getContext().getNextCommand();
                if (command != null) {
                    this.commandDispatcher.handle(command);
                    ++this.commandsHandled;
                    continue;
                }
                try {
                    long now = Time.monotonicNow();
                    if (this.nextHB.get() <= now) continue;
                    Thread.sleep(this.nextHB.get() - now + 1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
        };
        this.cmdProcessThread = this.getCommandHandlerThread(processCommandQueue);
        this.cmdProcessThread.start();
    }

    private Thread getCommandHandlerThread(Runnable processCommandQueue) {
        Thread handlerThread = new Thread(processCommandQueue);
        handlerThread.setDaemon(true);
        handlerThread.setName("Command processor thread");
        handlerThread.setUncaughtExceptionHandler((t, e) -> {
            LOG.error("Critical Error : Command processor thread encountered an error. Thread: {}", (Object)t.toString(), (Object)e);
            this.getCommandHandlerThread(processCommandQueue).start();
        });
        return handlerThread;
    }

    @VisibleForTesting
    public long getCommandHandled() {
        return this.commandsHandled;
    }

    @VisibleForTesting
    public CommandDispatcher getCommandDispatcher() {
        return this.commandDispatcher;
    }

    @VisibleForTesting
    public ReplicationSupervisor getSupervisor() {
        return this.supervisor;
    }

    public static enum DatanodeStates {
        INIT(1),
        RUNNING(2),
        SHUTDOWN(3);

        private final int value;

        private DatanodeStates(int value) {
            this.value = value;
        }

        public static DatanodeStates getInitState() {
            return INIT;
        }

        public static DatanodeStates getLastState() {
            return SHUTDOWN;
        }

        public int getValue() {
            return this.value;
        }

        public DatanodeStates getNextState() {
            if (this.value < DatanodeStates.getLastState().getValue()) {
                int stateValue = this.getValue() + 1;
                for (DatanodeStates iter : DatanodeStates.values()) {
                    if (stateValue != iter.getValue()) continue;
                    return iter;
                }
            }
            return DatanodeStates.getLastState();
        }

        public boolean isTransitionAllowedTo(DatanodeStates newState) {
            return newState.getValue() > this.getValue();
        }
    }
}

