/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.Principal;
import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.ArchiveOutputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.compressors.CompressorException;
import org.apache.commons.compress.compressors.CompressorOutputStream;
import org.apache.commons.compress.compressors.CompressorStreamFactory;
import org.apache.commons.compress.utils.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.hdds.utils.DBCheckpointMetrics;
import org.apache.hadoop.hdds.utils.db.DBCheckpoint;
import org.apache.hadoop.hdds.utils.db.DBStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DBCheckpointServlet
extends HttpServlet {
    private static final Logger LOG = LoggerFactory.getLogger(DBCheckpointServlet.class);
    private static final long serialVersionUID = 1L;
    private transient DBStore dbStore;
    private transient DBCheckpointMetrics dbMetrics;
    private boolean aclEnabled;
    private Collection<String> ozAdmins;

    public void initialize(DBStore store, DBCheckpointMetrics metrics, boolean omAclEnabled, Collection<String> ozoneAdmins) throws ServletException {
        this.dbStore = store;
        this.dbMetrics = metrics;
        if (this.dbStore == null) {
            LOG.error("Unable to set metadata snapshot request. DB Store is null");
        }
        this.aclEnabled = omAclEnabled;
        this.ozAdmins = ozoneAdmins;
    }

    private boolean hasPermission(String username) {
        if (this.aclEnabled) {
            return this.ozAdmins.contains("*") || this.ozAdmins.contains(username);
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void doGet(HttpServletRequest request, HttpServletResponse response) {
        LOG.info("Received request to obtain DB checkpoint snapshot");
        if (this.dbStore == null) {
            LOG.error("Unable to process metadata snapshot request. DB Store is null");
            response.setStatus(500);
            return;
        }
        if (this.aclEnabled) {
            Principal userPrincipal = request.getUserPrincipal();
            if (userPrincipal == null) {
                String remoteUser = request.getRemoteUser();
                LOG.error("Permission denied: Unauthorized access to /dbCheckpoint, no user principal found. Current login user is {}.", (Object)(remoteUser != null ? "'" + remoteUser + "'" : "UNKNOWN"));
                response.setStatus(403);
                return;
            }
            String userPrincipalName = userPrincipal.getName();
            if (!this.hasPermission(userPrincipalName)) {
                LOG.error("Permission denied: User principal '{}' does not have access to /dbCheckpoint.\nThis can happen when Ozone Manager is started with a different user.\n Please append '{}' to OM 'ozone.administrators' config and restart OM to grant current user access to this endpoint.", (Object)userPrincipalName, (Object)userPrincipalName);
                response.setStatus(403);
                return;
            }
            LOG.debug("Granted user principal '{}' access to /dbCheckpoint.", (Object)userPrincipalName);
        }
        DBCheckpoint checkpoint = null;
        try {
            boolean flush = false;
            String flushParam = request.getParameter("flushBeforeCheckpoint");
            if (StringUtils.isNotEmpty((CharSequence)flushParam)) {
                flush = Boolean.valueOf(flushParam);
            }
            if ((checkpoint = this.dbStore.getCheckpoint(flush)) == null || checkpoint.getCheckpointLocation() == null) {
                LOG.error("Unable to process metadata snapshot request. Checkpoint request returned null.");
                response.setStatus(500);
                return;
            }
            this.dbMetrics.setLastCheckpointCreationTimeTaken(checkpoint.checkpointCreationTimeTaken());
            Path file = checkpoint.getCheckpointLocation().getFileName();
            if (file == null) {
                return;
            }
            response.setContentType("application/x-tgz");
            response.setHeader("Content-Disposition", "attachment; filename=\"" + file.toString() + ".tgz\"");
            Instant start = Instant.now();
            DBCheckpointServlet.writeDBCheckpointToStream(checkpoint, (OutputStream)response.getOutputStream());
            Instant end = Instant.now();
            long duration = Duration.between(start, end).toMillis();
            LOG.info("Time taken to write the checkpoint to response output stream: {} milliseconds", (Object)duration);
            this.dbMetrics.setLastCheckpointStreamingTimeTaken(duration);
            this.dbMetrics.incNumCheckpoints();
        }
        catch (Exception e) {
            LOG.error("Unable to process metadata snapshot request. ", (Throwable)e);
            response.setStatus(500);
            this.dbMetrics.incNumCheckpointFails();
        }
        finally {
            if (checkpoint != null) {
                try {
                    checkpoint.cleanupCheckpoint();
                }
                catch (IOException e) {
                    LOG.error("Error trying to clean checkpoint at {} .", (Object)checkpoint.getCheckpointLocation().toString());
                }
            }
        }
    }

    public static void writeDBCheckpointToStream(DBCheckpoint checkpoint, OutputStream destination) throws IOException {
        try (CompressorOutputStream gzippedOut = new CompressorStreamFactory().createCompressorOutputStream("gz", destination);
             TarArchiveOutputStream archiveOutputStream = new TarArchiveOutputStream((OutputStream)gzippedOut);){
            Path checkpointPath = checkpoint.getCheckpointLocation();
            try (Stream<Path> files = Files.list(checkpointPath);){
                for (Path path : files.collect(Collectors.toList())) {
                    Path fileName;
                    if (path == null || (fileName = path.getFileName()) == null) continue;
                    DBCheckpointServlet.includeFile(path.toFile(), fileName.toString(), (ArchiveOutputStream)archiveOutputStream);
                }
            }
        }
        catch (CompressorException e) {
            throw new IOException("Can't compress the checkpoint: " + checkpoint.getCheckpointLocation(), e);
        }
    }

    private static void includeFile(File file, String entryName, ArchiveOutputStream archiveOutputStream) throws IOException {
        ArchiveEntry archiveEntry = archiveOutputStream.createArchiveEntry(file, entryName);
        archiveOutputStream.putArchiveEntry(archiveEntry);
        try (FileInputStream fis = new FileInputStream(file);){
            IOUtils.copy((InputStream)fis, (OutputStream)archiveOutputStream);
        }
        archiveOutputStream.closeArchiveEntry();
    }
}

