/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.storage;

import com.google.common.annotations.VisibleForTesting;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import org.apache.hadoop.fs.CanUnbuffer;
import org.apache.hadoop.fs.Seekable;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.XceiverClientFactory;
import org.apache.hadoop.hdds.scm.XceiverClientSpi;
import org.apache.hadoop.hdds.scm.client.HddsClientUtils;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.storage.ChunkInputStream;
import org.apache.hadoop.hdds.scm.storage.ContainerProtocolCalls;
import org.apache.hadoop.hdds.security.token.OzoneBlockTokenIdentifier;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.apache.hadoop.security.token.Token;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlockInputStream
extends InputStream
implements Seekable,
CanUnbuffer {
    private static final Logger LOG = LoggerFactory.getLogger(BlockInputStream.class);
    private static final int EOF = -1;
    private final BlockID blockID;
    private final long length;
    private Pipeline pipeline;
    private final Token<OzoneBlockTokenIdentifier> token;
    private final boolean verifyChecksum;
    private XceiverClientFactory xceiverClientFactory;
    private XceiverClientSpi xceiverClient;
    private boolean initialized = false;
    private final RetryPolicy retryPolicy = HddsClientUtils.createRetryPolicy(3, TimeUnit.SECONDS.toMillis(1L));
    private int retries;
    private List<ChunkInputStream> chunkStreams;
    private long[] chunkOffsets = null;
    private int chunkIndex;
    private long blockPosition = 0L;
    private int chunkIndexOfPrevPosition;
    private final Function<BlockID, Pipeline> refreshPipelineFunction;

    public BlockInputStream(BlockID blockId, long blockLen, Pipeline pipeline, Token<OzoneBlockTokenIdentifier> token, boolean verifyChecksum, XceiverClientFactory xceiverClientFactory, Function<BlockID, Pipeline> refreshPipelineFunction) {
        this.blockID = blockId;
        this.length = blockLen;
        this.pipeline = pipeline;
        this.token = token;
        this.verifyChecksum = verifyChecksum;
        this.xceiverClientFactory = xceiverClientFactory;
        this.refreshPipelineFunction = refreshPipelineFunction;
    }

    public BlockInputStream(BlockID blockId, long blockLen, Pipeline pipeline, Token<OzoneBlockTokenIdentifier> token, boolean verifyChecksum, XceiverClientFactory xceiverClientFactory) {
        this(blockId, blockLen, pipeline, token, verifyChecksum, xceiverClientFactory, null);
    }

    public synchronized void initialize() throws IOException {
        List<ContainerProtos.ChunkInfo> chunks;
        if (this.initialized) {
            return;
        }
        try {
            chunks = this.getChunkInfos();
        }
        catch (ContainerNotFoundException ioEx) {
            this.refreshPipeline((IOException)((Object)ioEx));
            chunks = this.getChunkInfos();
        }
        if (chunks != null && !chunks.isEmpty()) {
            this.chunkOffsets = new long[chunks.size()];
            long tempOffset = 0L;
            this.chunkStreams = new ArrayList<ChunkInputStream>(chunks.size());
            for (int i = 0; i < chunks.size(); ++i) {
                this.addStream(chunks.get(i));
                this.chunkOffsets[i] = tempOffset;
                tempOffset += chunks.get(i).getLen();
            }
            this.initialized = true;
            this.chunkIndex = 0;
            if (this.blockPosition > 0L) {
                this.seek(this.blockPosition);
            }
        }
    }

    private void refreshPipeline(IOException cause) throws IOException {
        Pipeline newPipeline;
        LOG.info("Unable to read information for block {} from pipeline {}: {}", new Object[]{this.blockID, this.pipeline.getId(), cause.getMessage()});
        if (this.refreshPipelineFunction != null) {
            LOG.debug("Re-fetching pipeline for block {}", (Object)this.blockID);
            newPipeline = this.refreshPipelineFunction.apply(this.blockID);
            if (newPipeline == null || newPipeline.sameDatanodes(this.pipeline)) {
                LOG.warn("No new pipeline for block {}", (Object)this.blockID);
                throw cause;
            }
        } else {
            throw cause;
        }
        LOG.debug("New pipeline got for block {}", (Object)this.blockID);
        this.pipeline = newPipeline;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected List<ContainerProtos.ChunkInfo> getChunkInfos() throws IOException {
        List chunks;
        if (this.pipeline.getType() != HddsProtos.ReplicationType.STAND_ALONE) {
            this.pipeline = Pipeline.newBuilder((Pipeline)this.pipeline).setType(HddsProtos.ReplicationType.STAND_ALONE).build();
        }
        this.acquireClient();
        boolean success = false;
        try {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Initializing BlockInputStream for get key to access {}", (Object)this.blockID.getContainerID());
            }
            ContainerProtos.DatanodeBlockID datanodeBlockID = this.blockID.getDatanodeBlockIDProtobuf();
            ContainerProtos.GetBlockResponseProto response = ContainerProtocolCalls.getBlock((XceiverClientSpi)this.xceiverClient, (ContainerProtos.DatanodeBlockID)datanodeBlockID, this.token);
            chunks = response.getBlockData().getChunksList();
            success = true;
        }
        finally {
            if (!success) {
                this.xceiverClientFactory.releaseClientForReadData(this.xceiverClient, false);
            }
        }
        return chunks;
    }

    protected void acquireClient() throws IOException {
        this.xceiverClient = this.xceiverClientFactory.acquireClientForReadData(this.pipeline);
    }

    protected synchronized void addStream(ContainerProtos.ChunkInfo chunkInfo) {
        this.chunkStreams.add(this.createChunkInputStream(chunkInfo));
    }

    protected ChunkInputStream createChunkInputStream(ContainerProtos.ChunkInfo chunkInfo) {
        return new ChunkInputStream(chunkInfo, this.blockID, this.xceiverClientFactory, () -> this.pipeline, this.verifyChecksum, this.token);
    }

    public synchronized long getRemaining() {
        return this.length - this.getPos();
    }

    @Override
    public synchronized int read() throws IOException {
        byte[] buf = new byte[1];
        if (this.read(buf, 0, 1) == -1) {
            return -1;
        }
        return Byte.toUnsignedInt(buf[0]);
    }

    @Override
    public synchronized int read(byte[] b, int off, int len) throws IOException {
        if (b == null) {
            throw new NullPointerException();
        }
        if (off < 0 || len < 0 || len > b.length - off) {
            throw new IndexOutOfBoundsException();
        }
        if (len == 0) {
            return 0;
        }
        if (!this.initialized) {
            this.initialize();
        }
        this.checkOpen();
        int totalReadLen = 0;
        while (len > 0) {
            int numBytesRead;
            if (this.chunkStreams.size() == 0 || this.chunkStreams.size() - 1 <= this.chunkIndex && this.chunkStreams.get(this.chunkIndex).getRemaining() == 0L) {
                return totalReadLen == 0 ? -1 : totalReadLen;
            }
            ChunkInputStream current = this.chunkStreams.get(this.chunkIndex);
            int numBytesToRead = Math.min(len, (int)current.getRemaining());
            try {
                numBytesRead = current.read(b, off, numBytesToRead);
                this.retries = 0;
            }
            catch (StorageContainerException e) {
                if (this.shouldRetryRead((IOException)((Object)e))) {
                    this.handleReadError((IOException)((Object)e));
                    continue;
                }
                throw e;
            }
            if (numBytesRead != numBytesToRead) {
                throw new IOException(String.format("Inconsistent read for chunkName=%s length=%d numBytesToRead= %d numBytesRead=%d", current.getChunkName(), current.getLength(), numBytesToRead, numBytesRead));
            }
            totalReadLen += numBytesRead;
            off += numBytesRead;
            len -= numBytesRead;
            if (current.getRemaining() > 0L || this.chunkIndex + 1 >= this.chunkStreams.size()) continue;
            ++this.chunkIndex;
        }
        return totalReadLen;
    }

    public synchronized void seek(long pos) throws IOException {
        if (!this.initialized) {
            this.blockPosition = pos;
            return;
        }
        this.checkOpen();
        if (pos < 0L || pos >= this.length) {
            if (pos == 0L) {
                return;
            }
            throw new EOFException("EOF encountered at pos: " + pos + " for block: " + this.blockID);
        }
        if (this.chunkIndex >= this.chunkStreams.size()) {
            this.chunkIndex = Arrays.binarySearch(this.chunkOffsets, pos);
        } else if (pos < this.chunkOffsets[this.chunkIndex]) {
            this.chunkIndex = Arrays.binarySearch(this.chunkOffsets, 0, this.chunkIndex, pos);
        } else if (pos >= this.chunkOffsets[this.chunkIndex] + this.chunkStreams.get(this.chunkIndex).getLength()) {
            this.chunkIndex = Arrays.binarySearch(this.chunkOffsets, this.chunkIndex + 1, this.chunkStreams.size(), pos);
        }
        if (this.chunkIndex < 0) {
            this.chunkIndex = -this.chunkIndex - 2;
        }
        this.chunkStreams.get(this.chunkIndexOfPrevPosition).resetPosition();
        for (int index = this.chunkIndex + 1; index < this.chunkStreams.size(); ++index) {
            this.chunkStreams.get(index).seek(0L);
        }
        this.chunkStreams.get(this.chunkIndex).seek(pos - this.chunkOffsets[this.chunkIndex]);
        this.chunkIndexOfPrevPosition = this.chunkIndex;
    }

    public synchronized long getPos() {
        if (this.length == 0L) {
            return 0L;
        }
        if (!this.initialized) {
            return this.blockPosition;
        }
        return this.chunkOffsets[this.chunkIndex] + this.chunkStreams.get(this.chunkIndex).getPos();
    }

    public boolean seekToNewSource(long targetPos) throws IOException {
        return false;
    }

    @Override
    public synchronized void close() {
        this.releaseClient();
        this.xceiverClientFactory = null;
        List<ChunkInputStream> inputStreams = this.chunkStreams;
        if (inputStreams != null) {
            for (ChunkInputStream is : inputStreams) {
                is.close();
            }
        }
    }

    private void releaseClient() {
        if (this.xceiverClientFactory != null && this.xceiverClient != null) {
            this.xceiverClientFactory.releaseClient(this.xceiverClient, false);
            this.xceiverClient = null;
        }
    }

    public synchronized void resetPosition() {
        this.blockPosition = 0L;
    }

    protected synchronized void checkOpen() throws IOException {
        if (this.xceiverClientFactory == null) {
            throw new IOException("BlockInputStream has been closed.");
        }
    }

    public BlockID getBlockID() {
        return this.blockID;
    }

    public long getLength() {
        return this.length;
    }

    @VisibleForTesting
    synchronized int getChunkIndex() {
        return this.chunkIndex;
    }

    @VisibleForTesting
    synchronized long getBlockPosition() {
        return this.blockPosition;
    }

    public void unbuffer() {
        this.storePosition();
        this.releaseClient();
        List<ChunkInputStream> inputStreams = this.chunkStreams;
        if (inputStreams != null) {
            for (ChunkInputStream is : inputStreams) {
                is.unbuffer();
            }
        }
    }

    private synchronized void storePosition() {
        this.blockPosition = this.getPos();
    }

    private boolean shouldRetryRead(IOException cause) throws IOException {
        RetryPolicy.RetryAction retryAction;
        try {
            retryAction = this.retryPolicy.shouldRetry((Exception)cause, ++this.retries, 0, true);
        }
        catch (IOException e) {
            throw e;
        }
        catch (Exception e) {
            throw new IOException(e);
        }
        return retryAction.action == RetryPolicy.RetryAction.RetryDecision.RETRY;
    }

    private void handleReadError(IOException cause) throws IOException {
        this.releaseClient();
        List<ChunkInputStream> inputStreams = this.chunkStreams;
        if (inputStreams != null) {
            for (ChunkInputStream is : inputStreams) {
                is.releaseClient();
            }
        }
        this.refreshPipeline(cause);
    }

    @VisibleForTesting
    public synchronized List<ChunkInputStream> getChunkStreams() {
        return this.chunkStreams;
    }
}

