/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.client.rpc;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.net.URI;
import java.security.InvalidKeyException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;
import org.apache.hadoop.crypto.CryptoInputStream;
import org.apache.hadoop.crypto.CryptoOutputStream;
import org.apache.hadoop.crypto.key.KeyProvider;
import org.apache.hadoop.fs.FileEncryptionInfo;
import org.apache.hadoop.hdds.client.ReplicationFactor;
import org.apache.hadoop.hdds.client.ReplicationType;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.StorageUnit;
import org.apache.hadoop.hdds.protocol.StorageType;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.OzoneClientConfig;
import org.apache.hadoop.hdds.scm.XceiverClientFactory;
import org.apache.hadoop.hdds.scm.XceiverClientManager;
import org.apache.hadoop.hdds.scm.client.HddsClientUtils;
import org.apache.hadoop.hdds.tracing.TracingUtil;
import org.apache.hadoop.hdds.utils.IOUtils;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.ozone.OzoneAcl;
import org.apache.hadoop.ozone.OzoneSecurityUtil;
import org.apache.hadoop.ozone.client.BucketArgs;
import org.apache.hadoop.ozone.client.OzoneBucket;
import org.apache.hadoop.ozone.client.OzoneKey;
import org.apache.hadoop.ozone.client.OzoneKeyDetails;
import org.apache.hadoop.ozone.client.OzoneKeyLocation;
import org.apache.hadoop.ozone.client.OzoneMultipartUpload;
import org.apache.hadoop.ozone.client.OzoneMultipartUploadList;
import org.apache.hadoop.ozone.client.OzoneMultipartUploadPartListParts;
import org.apache.hadoop.ozone.client.OzoneVolume;
import org.apache.hadoop.ozone.client.VolumeArgs;
import org.apache.hadoop.ozone.client.io.KeyInputStream;
import org.apache.hadoop.ozone.client.io.KeyOutputStream;
import org.apache.hadoop.ozone.client.io.LengthInputStream;
import org.apache.hadoop.ozone.client.io.MultipartCryptoKeyInputStream;
import org.apache.hadoop.ozone.client.io.OzoneCryptoInputStream;
import org.apache.hadoop.ozone.client.io.OzoneInputStream;
import org.apache.hadoop.ozone.client.io.OzoneOutputStream;
import org.apache.hadoop.ozone.client.protocol.ClientProtocol;
import org.apache.hadoop.ozone.client.rpc.OzoneKMSUtil;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.om.helpers.BucketEncryptionKeyInfo;
import org.apache.hadoop.ozone.om.helpers.OmBucketArgs;
import org.apache.hadoop.ozone.om.helpers.OmBucketInfo;
import org.apache.hadoop.ozone.om.helpers.OmDeleteKeys;
import org.apache.hadoop.ozone.om.helpers.OmKeyArgs;
import org.apache.hadoop.ozone.om.helpers.OmKeyInfo;
import org.apache.hadoop.ozone.om.helpers.OmMultipartInfo;
import org.apache.hadoop.ozone.om.helpers.OmMultipartUploadCompleteInfo;
import org.apache.hadoop.ozone.om.helpers.OmMultipartUploadCompleteList;
import org.apache.hadoop.ozone.om.helpers.OmMultipartUploadList;
import org.apache.hadoop.ozone.om.helpers.OmMultipartUploadListParts;
import org.apache.hadoop.ozone.om.helpers.OmPartInfo;
import org.apache.hadoop.ozone.om.helpers.OmRenameKeys;
import org.apache.hadoop.ozone.om.helpers.OmVolumeArgs;
import org.apache.hadoop.ozone.om.helpers.OpenKeySession;
import org.apache.hadoop.ozone.om.helpers.OzoneAclUtil;
import org.apache.hadoop.ozone.om.helpers.OzoneFileStatus;
import org.apache.hadoop.ozone.om.helpers.RepeatedOmKeyInfo;
import org.apache.hadoop.ozone.om.helpers.S3SecretValue;
import org.apache.hadoop.ozone.om.helpers.ServiceInfo;
import org.apache.hadoop.ozone.om.helpers.ServiceInfoEx;
import org.apache.hadoop.ozone.om.protocol.OzoneManagerProtocol;
import org.apache.hadoop.ozone.om.protocolPB.OmTransport;
import org.apache.hadoop.ozone.om.protocolPB.OmTransportFactory;
import org.apache.hadoop.ozone.om.protocolPB.OzoneManagerProtocolClientSideTranslatorPB;
import org.apache.hadoop.ozone.protocol.proto.OzoneManagerProtocolProtos;
import org.apache.hadoop.ozone.security.GDPRSymmetricKey;
import org.apache.hadoop.ozone.security.OzoneTokenIdentifier;
import org.apache.hadoop.ozone.security.acl.IAccessAuthorizer;
import org.apache.hadoop.ozone.security.acl.OzoneAclConfig;
import org.apache.hadoop.ozone.security.acl.OzoneObj;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.logging.log4j.util.Strings;
import org.apache.ratis.protocol.ClientId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RpcClient
implements ClientProtocol {
    private static final Logger LOG = LoggerFactory.getLogger(RpcClient.class);
    private final ConfigurationSource conf;
    private final OzoneManagerProtocol ozoneManagerClient;
    private final XceiverClientManager xceiverClientManager;
    private final int chunkSize;
    private final UserGroupInformation ugi;
    private final IAccessAuthorizer.ACLType userRights;
    private final IAccessAuthorizer.ACLType groupRights;
    private final long blockSize;
    private final ClientId clientId = ClientId.randomId();
    private final boolean unsafeByteBufferConversion;
    private Text dtService;
    private final boolean topologyAwareReadEnabled;
    private final boolean checkKeyNameEnabled;
    private final OzoneClientConfig clientConfig;

    public RpcClient(ConfigurationSource conf, String omServiceId) throws IOException {
        Preconditions.checkNotNull((Object)conf);
        this.conf = conf;
        this.ugi = UserGroupInformation.getCurrentUser();
        OzoneAclConfig aclConfig = (OzoneAclConfig)this.conf.getObject(OzoneAclConfig.class);
        this.userRights = aclConfig.getUserDefaultRights();
        this.groupRights = aclConfig.getGroupDefaultRights();
        this.clientConfig = (OzoneClientConfig)conf.getObject(OzoneClientConfig.class);
        OmTransport omTransport = OmTransportFactory.create((ConfigurationSource)conf, (UserGroupInformation)this.ugi, (String)omServiceId);
        this.ozoneManagerClient = (OzoneManagerProtocol)TracingUtil.createProxy((Object)new OzoneManagerProtocolClientSideTranslatorPB(omTransport, this.clientId.toString()), OzoneManagerProtocol.class, (ConfigurationSource)conf);
        this.dtService = omTransport.getDelegationTokenService();
        ServiceInfoEx serviceInfoEx = this.ozoneManagerClient.getServiceInfo();
        String caCertPem = null;
        if (OzoneSecurityUtil.isSecurityEnabled((ConfigurationSource)conf)) {
            caCertPem = serviceInfoEx.getCaCertificate();
        }
        this.xceiverClientManager = new XceiverClientManager(conf, (XceiverClientManager.ScmClientConfig)conf.getObject(XceiverClientManager.ScmClientConfig.class), caCertPem);
        int configuredChunkSize = (int)conf.getStorageSize("ozone.scm.chunk.size", "4MB", StorageUnit.BYTES);
        if (configuredChunkSize > 0x2000000) {
            LOG.warn("The chunk size ({}) is not allowed to be more than the maximum size ({}), resetting to the maximum size.", (Object)configuredChunkSize, (Object)0x2000000);
            this.chunkSize = 0x2000000;
        } else {
            this.chunkSize = configuredChunkSize;
        }
        this.blockSize = (long)conf.getStorageSize("ozone.scm.block.size", "256MB", StorageUnit.BYTES);
        this.unsafeByteBufferConversion = conf.getBoolean("ozone.UnsafeByteOperations.enabled", true);
        this.topologyAwareReadEnabled = conf.getBoolean("ozone.network.topology.aware.read", false);
        this.checkKeyNameEnabled = conf.getBoolean("ozone.om.keyname.character.check.enabled", false);
    }

    @Override
    public List<OzoneManagerProtocolProtos.OMRoleInfo> getOmRoleInfos() throws IOException {
        List serviceList = this.ozoneManagerClient.getServiceList();
        ArrayList<OzoneManagerProtocolProtos.OMRoleInfo> roleInfos = new ArrayList<OzoneManagerProtocolProtos.OMRoleInfo>();
        for (ServiceInfo serviceInfo : serviceList) {
            OzoneManagerProtocolProtos.OMRoleInfo omRoleInfo;
            if (!serviceInfo.getNodeType().equals((Object)HddsProtos.NodeType.OM) || (omRoleInfo = serviceInfo.getOmRoleInfo()) == null) continue;
            roleInfos.add(omRoleInfo);
        }
        return roleInfos;
    }

    @Override
    public void createVolume(String volumeName) throws IOException {
        this.createVolume(volumeName, VolumeArgs.newBuilder().build());
    }

    @Override
    public void createVolume(String volumeName, VolumeArgs volArgs) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        Preconditions.checkNotNull((Object)volArgs);
        RpcClient.verifyCountsQuota(volArgs.getQuotaInNamespace());
        RpcClient.verifySpaceQuota(volArgs.getQuotaInBytes());
        String admin = volArgs.getAdmin() == null ? this.ugi.getUserName() : volArgs.getAdmin();
        String owner = volArgs.getOwner() == null ? this.ugi.getUserName() : volArgs.getOwner();
        long quotaInNamespace = volArgs.getQuotaInNamespace();
        long quotaInBytes = volArgs.getQuotaInBytes();
        ArrayList<OzoneAcl> listOfAcls = new ArrayList<OzoneAcl>();
        listOfAcls.add(new OzoneAcl(IAccessAuthorizer.ACLIdentityType.USER, owner, this.userRights, OzoneAcl.AclScope.ACCESS));
        List<String> userGroups = Arrays.asList(UserGroupInformation.createRemoteUser((String)owner).getGroupNames());
        userGroups.stream().forEach(group -> listOfAcls.add(new OzoneAcl(IAccessAuthorizer.ACLIdentityType.GROUP, group, this.groupRights, OzoneAcl.AclScope.ACCESS)));
        if (volArgs.getAcls() != null) {
            listOfAcls.addAll(volArgs.getAcls());
        }
        OmVolumeArgs.Builder builder = OmVolumeArgs.newBuilder();
        builder.setVolume(volumeName);
        builder.setAdminName(admin);
        builder.setOwnerName(owner);
        builder.setQuotaInBytes(quotaInBytes);
        builder.setQuotaInNamespace(quotaInNamespace);
        builder.setUsedNamespace(0L);
        builder.addAllMetadata(volArgs.getMetadata());
        for (OzoneAcl ozoneAcl : listOfAcls.stream().distinct().collect(Collectors.toList())) {
            builder.addOzoneAcls(OzoneAcl.toProtobuf((OzoneAcl)ozoneAcl));
        }
        if (volArgs.getQuotaInBytes() == 0L) {
            LOG.info("Creating Volume: {}, with {} as owner.", (Object)volumeName, (Object)owner);
        } else {
            LOG.info("Creating Volume: {}, with {} as owner and space quota set to {} bytes, counts quota set to {}", new Object[]{volumeName, owner, quotaInBytes, quotaInNamespace});
        }
        this.ozoneManagerClient.createVolume(builder.build());
    }

    @Override
    public boolean setVolumeOwner(String volumeName, String owner) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        Preconditions.checkNotNull((Object)owner);
        return this.ozoneManagerClient.setOwner(volumeName, owner);
    }

    @Override
    public void setVolumeQuota(String volumeName, long quotaInNamespace, long quotaInBytes) throws IOException {
        HddsClientUtils.verifyResourceName((String)volumeName);
        RpcClient.verifyCountsQuota(quotaInNamespace);
        RpcClient.verifySpaceQuota(quotaInBytes);
        OmVolumeArgs omVolumeArgs = this.ozoneManagerClient.getVolumeInfo(volumeName);
        if (omVolumeArgs.getQuotaInNamespace() == -2L) {
            LOG.warn("Volume {} is created before version 1.1.0, usedNamespace may be inaccurate and it is not recommended to enable quota.", (Object)volumeName);
        }
        this.ozoneManagerClient.setQuota(volumeName, quotaInNamespace, quotaInBytes);
    }

    @Override
    public OzoneVolume getVolumeDetails(String volumeName) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        OmVolumeArgs volume = this.ozoneManagerClient.getVolumeInfo(volumeName);
        return new OzoneVolume(this.conf, this, volume.getVolume(), volume.getAdminName(), volume.getOwnerName(), volume.getQuotaInBytes(), volume.getQuotaInNamespace(), volume.getUsedNamespace(), volume.getCreationTime(), volume.getModificationTime(), volume.getAclMap().ozoneAclGetProtobuf().stream().map(OzoneAcl::fromProtobuf).collect(Collectors.toList()), volume.getMetadata());
    }

    @Override
    public boolean checkVolumeAccess(String volumeName, OzoneAcl acl) throws IOException {
        throw new UnsupportedOperationException("Not yet implemented.");
    }

    @Override
    public void deleteVolume(String volumeName) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        this.ozoneManagerClient.deleteVolume(volumeName);
    }

    @Override
    public List<OzoneVolume> listVolumes(String volumePrefix, String prevVolume, int maxListResult) throws IOException {
        List volumes = this.ozoneManagerClient.listAllVolumes(volumePrefix, prevVolume, maxListResult);
        return volumes.stream().map(volume -> new OzoneVolume(this.conf, this, volume.getVolume(), volume.getAdminName(), volume.getOwnerName(), volume.getQuotaInBytes(), volume.getQuotaInNamespace(), volume.getUsedNamespace(), volume.getCreationTime(), volume.getModificationTime(), volume.getAclMap().ozoneAclGetProtobuf().stream().map(OzoneAcl::fromProtobuf).collect(Collectors.toList()))).collect(Collectors.toList());
    }

    @Override
    public List<OzoneVolume> listVolumes(String user, String volumePrefix, String prevVolume, int maxListResult) throws IOException {
        List volumes = this.ozoneManagerClient.listVolumeByUser(user, volumePrefix, prevVolume, maxListResult);
        return volumes.stream().map(volume -> new OzoneVolume(this.conf, this, volume.getVolume(), volume.getAdminName(), volume.getOwnerName(), volume.getQuotaInBytes(), volume.getQuotaInNamespace(), volume.getUsedNamespace(), volume.getCreationTime(), volume.getModificationTime(), volume.getAclMap().ozoneAclGetProtobuf().stream().map(OzoneAcl::fromProtobuf).collect(Collectors.toList()), volume.getMetadata())).collect(Collectors.toList());
    }

    @Override
    public void createBucket(String volumeName, String bucketName) throws IOException {
        this.createBucket(volumeName, bucketName, BucketArgs.newBuilder().build());
    }

    @Override
    public void createBucket(String volumeName, String bucketName, BucketArgs bucketArgs) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        Preconditions.checkNotNull((Object)bucketArgs);
        RpcClient.verifyCountsQuota(bucketArgs.getQuotaInNamespace());
        RpcClient.verifySpaceQuota(bucketArgs.getQuotaInBytes());
        Boolean isVersionEnabled = bucketArgs.getVersioning() == null ? Boolean.FALSE : bucketArgs.getVersioning();
        StorageType storageType = bucketArgs.getStorageType() == null ? StorageType.DEFAULT : bucketArgs.getStorageType();
        BucketEncryptionKeyInfo bek = null;
        if (bucketArgs.getEncryptionKey() != null) {
            bek = new BucketEncryptionKeyInfo.Builder().setKeyName(bucketArgs.getEncryptionKey()).build();
        }
        List<OzoneAcl> listOfAcls = this.getAclList();
        if (bucketArgs.getAcls() != null) {
            listOfAcls.addAll(bucketArgs.getAcls());
        }
        OmBucketInfo.Builder builder = OmBucketInfo.newBuilder();
        builder.setVolumeName(volumeName).setBucketName(bucketName).setIsVersionEnabled(isVersionEnabled).addAllMetadata(bucketArgs.getMetadata()).setStorageType(storageType).setSourceVolume(bucketArgs.getSourceVolume()).setSourceBucket(bucketArgs.getSourceBucket()).setQuotaInBytes(bucketArgs.getQuotaInBytes()).setQuotaInNamespace(bucketArgs.getQuotaInNamespace()).setAcls(listOfAcls.stream().distinct().collect(Collectors.toList()));
        if (bek != null) {
            builder.setBucketEncryptionKey(bek);
        }
        LOG.info("Creating Bucket: {}/{}, with Versioning {} and Storage Type set to {} and Encryption set to {} ", new Object[]{volumeName, bucketName, isVersionEnabled, storageType, bek != null});
        this.ozoneManagerClient.createBucket(builder.build());
    }

    private static void verifyVolumeName(String volumeName) throws OMException {
        try {
            HddsClientUtils.verifyResourceName((String)volumeName);
        }
        catch (IllegalArgumentException e) {
            throw new OMException(e.getMessage(), OMException.ResultCodes.INVALID_VOLUME_NAME);
        }
    }

    private static void verifyBucketName(String bucketName) throws OMException {
        try {
            HddsClientUtils.verifyResourceName((String)bucketName);
        }
        catch (IllegalArgumentException e) {
            throw new OMException(e.getMessage(), OMException.ResultCodes.INVALID_BUCKET_NAME);
        }
    }

    private static void verifyCountsQuota(long quota) throws OMException {
        if (quota < -1L || quota == 0L) {
            throw new IllegalArgumentException("Invalid values for quota : counts quota is :" + quota + ".");
        }
    }

    private static void verifySpaceQuota(long quota) throws OMException {
        if (quota < -1L || quota == 0L) {
            throw new IllegalArgumentException("Invalid values for quota : space quota is :" + quota + ".");
        }
    }

    private List<OzoneAcl> getAclList() {
        return OzoneAclUtil.getAclList((String)this.ugi.getUserName(), (String[])this.ugi.getGroupNames(), (IAccessAuthorizer.ACLType)this.userRights, (IAccessAuthorizer.ACLType)this.groupRights);
    }

    @Override
    public Token<OzoneTokenIdentifier> getDelegationToken(Text renewer) throws IOException {
        Token token = this.ozoneManagerClient.getDelegationToken(renewer);
        if (token != null) {
            token.setService(this.dtService);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Created token {} for dtService {}", (Object)token, (Object)this.dtService);
            }
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("Cannot get ozone delegation token for renewer {} to access service {}", (Object)renewer, (Object)this.dtService);
        }
        return token;
    }

    @Override
    public long renewDelegationToken(Token<OzoneTokenIdentifier> token) throws IOException {
        return this.ozoneManagerClient.renewDelegationToken(token);
    }

    @Override
    public void cancelDelegationToken(Token<OzoneTokenIdentifier> token) throws IOException {
        this.ozoneManagerClient.cancelDelegationToken(token);
    }

    @Override
    public S3SecretValue getS3Secret(String kerberosID) throws IOException {
        Preconditions.checkArgument((boolean)Strings.isNotBlank((String)kerberosID), (Object)"kerberosID cannot be null or empty.");
        return this.ozoneManagerClient.getS3Secret(kerberosID);
    }

    @Override
    public void setBucketVersioning(String volumeName, String bucketName, Boolean versioning) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        Preconditions.checkNotNull((Object)versioning);
        OmBucketArgs.Builder builder = OmBucketArgs.newBuilder();
        builder.setVolumeName(volumeName).setBucketName(bucketName).setIsVersionEnabled(versioning);
        this.ozoneManagerClient.setBucketProperty(builder.build());
    }

    @Override
    public void setBucketStorageType(String volumeName, String bucketName, StorageType storageType) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        Preconditions.checkNotNull((Object)storageType);
        OmBucketArgs.Builder builder = OmBucketArgs.newBuilder();
        builder.setVolumeName(volumeName).setBucketName(bucketName).setStorageType(storageType);
        this.ozoneManagerClient.setBucketProperty(builder.build());
    }

    @Override
    public void setBucketQuota(String volumeName, String bucketName, long quotaInNamespace, long quotaInBytes) throws IOException {
        HddsClientUtils.verifyResourceName((String)bucketName);
        HddsClientUtils.verifyResourceName((String)volumeName);
        RpcClient.verifyCountsQuota(quotaInNamespace);
        RpcClient.verifySpaceQuota(quotaInBytes);
        OmBucketArgs.Builder builder = OmBucketArgs.newBuilder();
        builder.setVolumeName(volumeName).setBucketName(bucketName).setQuotaInBytes(quotaInBytes).setQuotaInNamespace(quotaInNamespace);
        OmBucketInfo omBucketInfo = this.ozoneManagerClient.getBucketInfo(volumeName, bucketName);
        if (omBucketInfo.getQuotaInNamespace() == -2L || omBucketInfo.getUsedBytes() == -2L) {
            LOG.warn("Bucket {} is created before version 1.1.0, usedBytes or usedNamespace may be inaccurate and it is not recommended to enable quota.", (Object)bucketName);
        }
        this.ozoneManagerClient.setBucketProperty(builder.build());
    }

    @Override
    public void deleteBucket(String volumeName, String bucketName) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        this.ozoneManagerClient.deleteBucket(volumeName, bucketName);
    }

    @Override
    public void checkBucketAccess(String volumeName, String bucketName) throws IOException {
    }

    @Override
    public OzoneBucket getBucketDetails(String volumeName, String bucketName) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        OmBucketInfo bucketInfo = this.ozoneManagerClient.getBucketInfo(volumeName, bucketName);
        return new OzoneBucket(this.conf, this, bucketInfo.getVolumeName(), bucketInfo.getBucketName(), bucketInfo.getStorageType(), bucketInfo.getIsVersionEnabled(), bucketInfo.getCreationTime(), bucketInfo.getModificationTime(), bucketInfo.getMetadata(), bucketInfo.getEncryptionKeyInfo() != null ? bucketInfo.getEncryptionKeyInfo().getKeyName() : null, bucketInfo.getSourceVolume(), bucketInfo.getSourceBucket(), bucketInfo.getUsedBytes(), bucketInfo.getUsedNamespace(), bucketInfo.getQuotaInBytes(), bucketInfo.getQuotaInNamespace());
    }

    @Override
    public List<OzoneBucket> listBuckets(String volumeName, String bucketPrefix, String prevBucket, int maxListResult) throws IOException {
        List buckets = this.ozoneManagerClient.listBuckets(volumeName, prevBucket, bucketPrefix, maxListResult);
        return buckets.stream().map(bucket -> new OzoneBucket(this.conf, this, bucket.getVolumeName(), bucket.getBucketName(), bucket.getStorageType(), bucket.getIsVersionEnabled(), bucket.getCreationTime(), bucket.getModificationTime(), bucket.getMetadata(), bucket.getEncryptionKeyInfo() != null ? bucket.getEncryptionKeyInfo().getKeyName() : null, bucket.getSourceVolume(), bucket.getSourceBucket(), bucket.getUsedBytes(), bucket.getUsedNamespace(), bucket.getQuotaInBytes(), bucket.getQuotaInNamespace())).collect(Collectors.toList());
    }

    @Override
    public OzoneOutputStream createKey(String volumeName, String bucketName, String keyName, long size, ReplicationType type, ReplicationFactor factor, Map<String, String> metadata) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        if (this.checkKeyNameEnabled) {
            HddsClientUtils.verifyKeyName((String)keyName);
        }
        HddsClientUtils.checkNotNull((Object[])new Serializable[]{keyName, type, factor});
        String requestId = UUID.randomUUID().toString();
        OmKeyArgs.Builder builder = new OmKeyArgs.Builder().setVolumeName(volumeName).setBucketName(bucketName).setKeyName(keyName).setDataSize(size).setType(HddsProtos.ReplicationType.valueOf((String)type.toString())).setFactor(HddsProtos.ReplicationFactor.valueOf((int)factor.getValue())).addAllMetadata(metadata).setAcls(this.getAclList());
        if (Boolean.parseBoolean(metadata.get("gdprEnabled"))) {
            try {
                GDPRSymmetricKey gKey = new GDPRSymmetricKey(new SecureRandom());
                builder.addAllMetadata(gKey.getKeyDetails());
            }
            catch (Exception e) {
                if (e instanceof InvalidKeyException && e.getMessage().contains("Illegal key size or default parameters")) {
                    LOG.error("Missing Unlimited Strength Policy jars. Please install Java Cryptography Extension (JCE) Unlimited Strength Jurisdiction Policy Files");
                }
                throw new IOException(e);
            }
        }
        OpenKeySession openKey = this.ozoneManagerClient.openKey(builder.build());
        return this.createOutputStream(openKey, requestId, type, factor);
    }

    private KeyProvider.KeyVersion getDEK(FileEncryptionInfo feInfo) throws IOException {
        OzoneKMSUtil.checkCryptoProtocolVersion(feInfo);
        KeyProvider.KeyVersion decrypted = OzoneKMSUtil.decryptEncryptedDataEncryptionKey(feInfo, this.getKeyProvider());
        return decrypted;
    }

    @Override
    public OzoneInputStream getKey(String volumeName, String bucketName, String keyName) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        Preconditions.checkNotNull((Object)keyName);
        OmKeyArgs keyArgs = new OmKeyArgs.Builder().setVolumeName(volumeName).setBucketName(bucketName).setKeyName(keyName).setRefreshPipeline(true).setSortDatanodesInPipeline(this.topologyAwareReadEnabled).build();
        OmKeyInfo keyInfo = this.ozoneManagerClient.lookupKey(keyArgs);
        return this.getInputStreamWithRetryFunction(keyInfo);
    }

    @Override
    public void deleteKey(String volumeName, String bucketName, String keyName) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        Preconditions.checkNotNull((Object)keyName);
        OmKeyArgs keyArgs = new OmKeyArgs.Builder().setVolumeName(volumeName).setBucketName(bucketName).setKeyName(keyName).build();
        this.ozoneManagerClient.deleteKey(keyArgs);
    }

    @Override
    public void deleteKeys(String volumeName, String bucketName, List<String> keyNameList) throws IOException {
        HddsClientUtils.verifyResourceName((String[])new String[]{volumeName, bucketName});
        Preconditions.checkNotNull(keyNameList);
        OmDeleteKeys omDeleteKeys = new OmDeleteKeys(volumeName, bucketName, keyNameList);
        this.ozoneManagerClient.deleteKeys(omDeleteKeys);
    }

    @Override
    public void renameKey(String volumeName, String bucketName, String fromKeyName, String toKeyName) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        if (this.checkKeyNameEnabled) {
            HddsClientUtils.verifyKeyName((String)toKeyName);
        }
        HddsClientUtils.checkNotNull((Object[])new String[]{fromKeyName, toKeyName});
        OmKeyArgs keyArgs = new OmKeyArgs.Builder().setVolumeName(volumeName).setBucketName(bucketName).setKeyName(fromKeyName).build();
        this.ozoneManagerClient.renameKey(keyArgs, toKeyName);
    }

    @Override
    public void renameKeys(String volumeName, String bucketName, Map<String, String> keyMap) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        HddsClientUtils.checkNotNull((Object[])new Map[]{keyMap});
        OmRenameKeys omRenameKeys = new OmRenameKeys(volumeName, bucketName, keyMap, null);
        this.ozoneManagerClient.renameKeys(omRenameKeys);
    }

    @Override
    public List<OzoneKey> listKeys(String volumeName, String bucketName, String keyPrefix, String prevKey, int maxListResult) throws IOException {
        List keys = this.ozoneManagerClient.listKeys(volumeName, bucketName, prevKey, keyPrefix, maxListResult);
        return keys.stream().map(key -> new OzoneKey(key.getVolumeName(), key.getBucketName(), key.getKeyName(), key.getDataSize(), key.getCreationTime(), key.getModificationTime(), ReplicationType.valueOf((String)key.getType().toString()), key.getFactor().getNumber())).collect(Collectors.toList());
    }

    @Override
    public List<RepeatedOmKeyInfo> listTrash(String volumeName, String bucketName, String startKeyName, String keyPrefix, int maxKeys) throws IOException {
        Preconditions.checkNotNull((Object)volumeName);
        Preconditions.checkNotNull((Object)bucketName);
        return this.ozoneManagerClient.listTrash(volumeName, bucketName, startKeyName, keyPrefix, maxKeys);
    }

    @Override
    public boolean recoverTrash(String volumeName, String bucketName, String keyName, String destinationBucket) throws IOException {
        return this.ozoneManagerClient.recoverTrash(volumeName, bucketName, keyName, destinationBucket);
    }

    @Override
    public OzoneKeyDetails getKeyDetails(String volumeName, String bucketName, String keyName) throws IOException {
        Preconditions.checkNotNull((Object)volumeName);
        Preconditions.checkNotNull((Object)bucketName);
        Preconditions.checkNotNull((Object)keyName);
        OmKeyArgs keyArgs = new OmKeyArgs.Builder().setVolumeName(volumeName).setBucketName(bucketName).setKeyName(keyName).setRefreshPipeline(true).setSortDatanodesInPipeline(this.topologyAwareReadEnabled).build();
        OmKeyInfo keyInfo = this.ozoneManagerClient.lookupKey(keyArgs);
        ArrayList<OzoneKeyLocation> ozoneKeyLocations = new ArrayList<OzoneKeyLocation>();
        keyInfo.getLatestVersionLocations().getBlocksLatestVersionOnly().forEach(a -> ozoneKeyLocations.add(new OzoneKeyLocation(a.getContainerID(), a.getLocalID(), a.getLength(), a.getOffset())));
        return new OzoneKeyDetails(keyInfo.getVolumeName(), keyInfo.getBucketName(), keyInfo.getKeyName(), keyInfo.getDataSize(), keyInfo.getCreationTime(), keyInfo.getModificationTime(), ozoneKeyLocations, ReplicationType.valueOf((String)keyInfo.getType().toString()), keyInfo.getMetadata(), keyInfo.getFileEncryptionInfo(), keyInfo.getFactor().getNumber());
    }

    @Override
    public void close() throws IOException {
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{this.ozoneManagerClient, this.xceiverClientManager});
    }

    @Override
    public OmMultipartInfo initiateMultipartUpload(String volumeName, String bucketName, String keyName, ReplicationType type, ReplicationFactor factor) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        HddsClientUtils.checkNotNull((Object[])new Serializable[]{keyName, type, factor});
        OmKeyArgs keyArgs = new OmKeyArgs.Builder().setVolumeName(volumeName).setBucketName(bucketName).setKeyName(keyName).setType(HddsProtos.ReplicationType.valueOf((String)type.toString())).setFactor(HddsProtos.ReplicationFactor.valueOf((int)factor.getValue())).setAcls(this.getAclList()).build();
        OmMultipartInfo multipartInfo = this.ozoneManagerClient.initiateMultipartUpload(keyArgs);
        return multipartInfo;
    }

    @Override
    public OzoneOutputStream createMultipartKey(String volumeName, String bucketName, String keyName, long size, int partNumber, String uploadID) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        if (this.checkKeyNameEnabled) {
            HddsClientUtils.verifyKeyName((String)keyName);
        }
        HddsClientUtils.checkNotNull((Object[])new String[]{keyName, uploadID});
        Preconditions.checkArgument((partNumber > 0 && partNumber <= 10000 ? 1 : 0) != 0, (Object)"Part number should be greater than zero and less than or equal to 10000");
        Preconditions.checkArgument((size >= 0L ? 1 : 0) != 0, (Object)"size should be greater than or equal to zero");
        String requestId = UUID.randomUUID().toString();
        OmKeyArgs keyArgs = new OmKeyArgs.Builder().setVolumeName(volumeName).setBucketName(bucketName).setKeyName(keyName).setDataSize(size).setIsMultipartKey(true).setMultipartUploadID(uploadID).setMultipartUploadPartNumber(partNumber).setAcls(this.getAclList()).build();
        OpenKeySession openKey = this.ozoneManagerClient.openKey(keyArgs);
        KeyOutputStream keyOutputStream = new KeyOutputStream.Builder().setHandler(openKey).setXceiverClientManager(this.xceiverClientManager).setOmClient(this.ozoneManagerClient).setRequestID(requestId).setType(openKey.getKeyInfo().getType()).setFactor(openKey.getKeyInfo().getFactor()).setMultipartNumber(partNumber).setMultipartUploadID(uploadID).setIsMultipartKey(true).enableUnsafeByteBufferConversion(this.unsafeByteBufferConversion).setConfig(this.clientConfig).build();
        keyOutputStream.addPreallocateBlocks(openKey.getKeyInfo().getLatestVersionLocations(), openKey.getOpenVersion());
        FileEncryptionInfo feInfo = keyOutputStream.getFileEncryptionInfo();
        if (feInfo != null) {
            KeyProvider.KeyVersion decrypted = this.getDEK(feInfo);
            CryptoOutputStream cryptoOut = new CryptoOutputStream((OutputStream)keyOutputStream, OzoneKMSUtil.getCryptoCodec(this.conf, feInfo), decrypted.getMaterial(), feInfo.getIV());
            return new OzoneOutputStream((OutputStream)cryptoOut);
        }
        return new OzoneOutputStream(keyOutputStream);
    }

    @Override
    public OmMultipartUploadCompleteInfo completeMultipartUpload(String volumeName, String bucketName, String keyName, String uploadID, Map<Integer, String> partsMap) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        HddsClientUtils.checkNotNull((Object[])new String[]{keyName, uploadID});
        OmKeyArgs keyArgs = new OmKeyArgs.Builder().setVolumeName(volumeName).setBucketName(bucketName).setKeyName(keyName).setMultipartUploadID(uploadID).setAcls(this.getAclList()).build();
        OmMultipartUploadCompleteList omMultipartUploadCompleteList = new OmMultipartUploadCompleteList(partsMap);
        OmMultipartUploadCompleteInfo omMultipartUploadCompleteInfo = this.ozoneManagerClient.completeMultipartUpload(keyArgs, omMultipartUploadCompleteList);
        return omMultipartUploadCompleteInfo;
    }

    @Override
    public void abortMultipartUpload(String volumeName, String bucketName, String keyName, String uploadID) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        HddsClientUtils.checkNotNull((Object[])new String[]{keyName, uploadID});
        OmKeyArgs omKeyArgs = new OmKeyArgs.Builder().setVolumeName(volumeName).setBucketName(bucketName).setKeyName(keyName).setMultipartUploadID(uploadID).build();
        this.ozoneManagerClient.abortMultipartUpload(omKeyArgs);
    }

    @Override
    public OzoneMultipartUploadPartListParts listParts(String volumeName, String bucketName, String keyName, String uploadID, int partNumberMarker, int maxParts) throws IOException {
        RpcClient.verifyVolumeName(volumeName);
        RpcClient.verifyBucketName(bucketName);
        HddsClientUtils.checkNotNull((Object[])new String[]{uploadID});
        Preconditions.checkArgument((maxParts > 0 ? 1 : 0) != 0, (Object)"Max Parts Should be greater than zero");
        Preconditions.checkArgument((partNumberMarker >= 0 ? 1 : 0) != 0, (Object)"Part Number Marker Should be greater than or equal to zero, as part numbers starts from 1 and ranges till 10000");
        OmMultipartUploadListParts omMultipartUploadListParts = this.ozoneManagerClient.listParts(volumeName, bucketName, keyName, uploadID, partNumberMarker, maxParts);
        OzoneMultipartUploadPartListParts ozoneMultipartUploadPartListParts = new OzoneMultipartUploadPartListParts(ReplicationType.fromProto((HddsProtos.ReplicationType)omMultipartUploadListParts.getReplicationType()), ReplicationFactor.fromProto((HddsProtos.ReplicationFactor)omMultipartUploadListParts.getReplicationFactor()), omMultipartUploadListParts.getNextPartNumberMarker(), omMultipartUploadListParts.isTruncated());
        for (OmPartInfo omPartInfo : omMultipartUploadListParts.getPartInfoList()) {
            ozoneMultipartUploadPartListParts.addPart(new OzoneMultipartUploadPartListParts.PartInfo(omPartInfo.getPartNumber(), omPartInfo.getPartName(), omPartInfo.getModificationTime(), omPartInfo.getSize()));
        }
        return ozoneMultipartUploadPartListParts;
    }

    @Override
    public OzoneMultipartUploadList listMultipartUploads(String volumeName, String bucketName, String prefix) throws IOException {
        OmMultipartUploadList omMultipartUploadList = this.ozoneManagerClient.listMultipartUploads(volumeName, bucketName, prefix);
        List<OzoneMultipartUpload> uploads = omMultipartUploadList.getUploads().stream().map(upload -> new OzoneMultipartUpload(upload.getVolumeName(), upload.getBucketName(), upload.getKeyName(), upload.getUploadId(), upload.getCreationTime(), ReplicationType.fromProto((HddsProtos.ReplicationType)upload.getReplicationType()), ReplicationFactor.fromProto((HddsProtos.ReplicationFactor)upload.getReplicationFactor()))).collect(Collectors.toList());
        OzoneMultipartUploadList result = new OzoneMultipartUploadList(uploads);
        return result;
    }

    @Override
    public OzoneFileStatus getOzoneFileStatus(String volumeName, String bucketName, String keyName) throws IOException {
        OmKeyArgs keyArgs = new OmKeyArgs.Builder().setVolumeName(volumeName).setBucketName(bucketName).setKeyName(keyName).setRefreshPipeline(true).setSortDatanodesInPipeline(this.topologyAwareReadEnabled).build();
        return this.ozoneManagerClient.getFileStatus(keyArgs);
    }

    @Override
    public void createDirectory(String volumeName, String bucketName, String keyName) throws IOException {
        OmKeyArgs keyArgs = new OmKeyArgs.Builder().setVolumeName(volumeName).setBucketName(bucketName).setKeyName(keyName).setAcls(this.getAclList()).build();
        this.ozoneManagerClient.createDirectory(keyArgs);
    }

    @Override
    public OzoneInputStream readFile(String volumeName, String bucketName, String keyName) throws IOException {
        OmKeyArgs keyArgs = new OmKeyArgs.Builder().setVolumeName(volumeName).setBucketName(bucketName).setKeyName(keyName).setSortDatanodesInPipeline(this.topologyAwareReadEnabled).build();
        OmKeyInfo keyInfo = this.ozoneManagerClient.lookupFile(keyArgs);
        return this.getInputStreamWithRetryFunction(keyInfo);
    }

    private OzoneInputStream getInputStreamWithRetryFunction(OmKeyInfo keyInfo) throws IOException {
        return this.createInputStream(keyInfo, omKeyInfo -> {
            try {
                OmKeyArgs omKeyArgs = new OmKeyArgs.Builder().setVolumeName(omKeyInfo.getVolumeName()).setBucketName(omKeyInfo.getBucketName()).setKeyName(omKeyInfo.getKeyName()).setRefreshPipeline(true).setSortDatanodesInPipeline(this.topologyAwareReadEnabled).build();
                return this.ozoneManagerClient.lookupKey(omKeyArgs);
            }
            catch (IOException e) {
                LOG.error("Unable to lookup key {} on retry.", (Object)keyInfo.getKeyName(), (Object)e);
                return null;
            }
        });
    }

    @Override
    public OzoneOutputStream createFile(String volumeName, String bucketName, String keyName, long size, ReplicationType type, ReplicationFactor factor, boolean overWrite, boolean recursive) throws IOException {
        OmKeyArgs keyArgs = new OmKeyArgs.Builder().setVolumeName(volumeName).setBucketName(bucketName).setKeyName(keyName).setDataSize(size).setType(HddsProtos.ReplicationType.valueOf((String)type.name())).setFactor(HddsProtos.ReplicationFactor.valueOf((int)factor.getValue())).setAcls(this.getAclList()).build();
        OpenKeySession keySession = this.ozoneManagerClient.createFile(keyArgs, overWrite, recursive);
        return this.createOutputStream(keySession, UUID.randomUUID().toString(), type, factor);
    }

    @Override
    public List<OzoneFileStatus> listStatus(String volumeName, String bucketName, String keyName, boolean recursive, String startKey, long numEntries) throws IOException {
        OmKeyArgs keyArgs = new OmKeyArgs.Builder().setVolumeName(volumeName).setBucketName(bucketName).setKeyName(keyName).setRefreshPipeline(true).setSortDatanodesInPipeline(this.topologyAwareReadEnabled).build();
        return this.ozoneManagerClient.listStatus(keyArgs, recursive, startKey, numEntries);
    }

    @Override
    public boolean addAcl(OzoneObj obj, OzoneAcl acl) throws IOException {
        return this.ozoneManagerClient.addAcl(obj, acl);
    }

    @Override
    public boolean removeAcl(OzoneObj obj, OzoneAcl acl) throws IOException {
        return this.ozoneManagerClient.removeAcl(obj, acl);
    }

    @Override
    public boolean setAcl(OzoneObj obj, List<OzoneAcl> acls) throws IOException {
        return this.ozoneManagerClient.setAcl(obj, acls);
    }

    @Override
    public List<OzoneAcl> getAcl(OzoneObj obj) throws IOException {
        return this.ozoneManagerClient.getAcl(obj);
    }

    private OzoneInputStream createInputStream(OmKeyInfo keyInfo, Function<OmKeyInfo, OmKeyInfo> retryFunction) throws IOException {
        FileEncryptionInfo feInfo = keyInfo.getFileEncryptionInfo();
        if (feInfo == null) {
            LengthInputStream lengthInputStream = KeyInputStream.getFromOmKeyInfo(keyInfo, (XceiverClientFactory)this.xceiverClientManager, this.clientConfig.isChecksumVerify(), retryFunction);
            try {
                Map keyInfoMetadata = keyInfo.getMetadata();
                if (Boolean.valueOf((String)keyInfoMetadata.get("gdprEnabled")).booleanValue()) {
                    GDPRSymmetricKey gk = new GDPRSymmetricKey((String)keyInfoMetadata.get("secret"), (String)keyInfoMetadata.get("algorithm"));
                    gk.getCipher().init(2, gk.getSecretKey());
                    return new OzoneInputStream(new CipherInputStream((InputStream)lengthInputStream, gk.getCipher()));
                }
            }
            catch (Exception ex) {
                throw new IOException(ex);
            }
            return new OzoneInputStream(lengthInputStream.getWrappedStream());
        }
        if (!keyInfo.getLatestVersionLocations().isMultipartKey()) {
            LengthInputStream lengthInputStream = KeyInputStream.getFromOmKeyInfo(keyInfo, (XceiverClientFactory)this.xceiverClientManager, this.clientConfig.isChecksumVerify(), retryFunction);
            KeyProvider.KeyVersion decrypted = this.getDEK(feInfo);
            CryptoInputStream cryptoIn = new CryptoInputStream(lengthInputStream.getWrappedStream(), OzoneKMSUtil.getCryptoCodec(this.conf, feInfo), decrypted.getMaterial(), feInfo.getIV());
            return new OzoneInputStream((InputStream)cryptoIn);
        }
        List<LengthInputStream> lengthInputStreams = KeyInputStream.getStreamsFromKeyInfo(keyInfo, (XceiverClientFactory)this.xceiverClientManager, this.clientConfig.isChecksumVerify(), retryFunction);
        KeyProvider.KeyVersion decrypted = this.getDEK(feInfo);
        ArrayList<OzoneCryptoInputStream> cryptoInputStreams = new ArrayList<OzoneCryptoInputStream>();
        for (LengthInputStream lengthInputStream : lengthInputStreams) {
            OzoneCryptoInputStream ozoneCryptoInputStream = new OzoneCryptoInputStream(lengthInputStream, OzoneKMSUtil.getCryptoCodec(this.conf, feInfo), decrypted.getMaterial(), feInfo.getIV());
            cryptoInputStreams.add(ozoneCryptoInputStream);
        }
        return new MultipartCryptoKeyInputStream(keyInfo.getKeyName(), cryptoInputStreams);
    }

    private OzoneOutputStream createOutputStream(OpenKeySession openKey, String requestId, ReplicationType type, ReplicationFactor factor) throws IOException {
        KeyOutputStream keyOutputStream = new KeyOutputStream.Builder().setHandler(openKey).setXceiverClientManager(this.xceiverClientManager).setOmClient(this.ozoneManagerClient).setRequestID(requestId).setType(HddsProtos.ReplicationType.valueOf((String)type.toString())).setFactor(HddsProtos.ReplicationFactor.valueOf((int)factor.getValue())).enableUnsafeByteBufferConversion(this.unsafeByteBufferConversion).setConfig(this.clientConfig).build();
        keyOutputStream.addPreallocateBlocks(openKey.getKeyInfo().getLatestVersionLocations(), openKey.getOpenVersion());
        FileEncryptionInfo feInfo = keyOutputStream.getFileEncryptionInfo();
        if (feInfo != null) {
            KeyProvider.KeyVersion decrypted = this.getDEK(feInfo);
            CryptoOutputStream cryptoOut = new CryptoOutputStream((OutputStream)keyOutputStream, OzoneKMSUtil.getCryptoCodec(this.conf, feInfo), decrypted.getMaterial(), feInfo.getIV());
            return new OzoneOutputStream((OutputStream)cryptoOut);
        }
        try {
            Map openKeyMetadata = openKey.getKeyInfo().getMetadata();
            if (Boolean.valueOf((String)openKeyMetadata.get("gdprEnabled")).booleanValue()) {
                GDPRSymmetricKey gk = new GDPRSymmetricKey((String)openKeyMetadata.get("secret"), (String)openKeyMetadata.get("algorithm"));
                gk.getCipher().init(1, gk.getSecretKey());
                return new OzoneOutputStream(new CipherOutputStream(keyOutputStream, gk.getCipher()));
            }
        }
        catch (Exception ex) {
            throw new IOException(ex);
        }
        return new OzoneOutputStream(keyOutputStream);
    }

    @Override
    public KeyProvider getKeyProvider() throws IOException {
        return OzoneKMSUtil.getKeyProvider(this.conf, this.getKeyProviderUri());
    }

    @Override
    public URI getKeyProviderUri() throws IOException {
        return OzoneKMSUtil.getKeyProviderUri(this.ugi, null, null, this.conf);
    }

    @Override
    public String getCanonicalServiceName() {
        return this.dtService != null ? this.dtService.toString() : null;
    }

    @Override
    @VisibleForTesting
    public OzoneManagerProtocol getOzoneManagerClient() {
        return this.ozoneManagerClient;
    }
}

