/****************************************************************************
 * libs/libnx/nxglib/nxglib_circlepts.c
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.  The
 * ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 ****************************************************************************/

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>

#include <string.h>
#include <errno.h>
#include <fixedmath.h>

#include <nuttx/nx/nxglib.h>

/****************************************************************************
 * Pre-Processor Definitions
 ****************************************************************************/

/* Trigonometry */

#define SIN_0p0        0         /* sin(0) = 0 */
#define COS_0p0        1         /* cos(0) = 1 */
#define SIN_22p5       25080     /* sin(22.5) = 25080 / 65536 */
#define COS_22p5       60547     /* cos(22.5) = 60547 / 65536 */
#define SIN_45p0       46341     /* sin(45) = 46341 / 65536 */
#define COS_45p0       SIN_45p0  /* cos(45) = sin(45) */

/* Named indices into the 16 circle points generated by nxgl_circlepts */

#define POINT_0p0      0
#define POINT_22p5     1
#define POINT_45p0     2
#define POINT_67p5     3
#define POINT_90p0     4
#define POINT_112p5    5
#define POINT_135p0    6
#define POINT_157p5    7
#define POINT_180p0    8
#define POINT_202p5    9
#define POINT_225p0    10
#define POINT_247p5    11
#define POINT_270p0    12
#define POINT_292p5    13
#define POINT_315p0    14
#define POINT_337p5    15
#define NCIRCLE_POINTS 16

/****************************************************************************
 * Public Functions
 ****************************************************************************/

/****************************************************************************
 * Name: nxgl_circlepts
 *
 * Description:
 *   Given a description of a circle, return a set of 16 points on the
 *   circumference of the circle.  These points may then be used by
 *   nx_drawcircle() or related APIs to draw a circle outline.
 *
 * Input Parameters:
 *   center - A pointer to the point that is the center of the circle
 *   radius - The radius of the circle in pixels.
 *   circle - A pointer the first entry in an array of 16 points where the
 *            circle points will be returned.
 *
 * Returned Value:
 *   None
 *
 ****************************************************************************/

void nxgl_circlepts(FAR const struct nxgl_point_s *center,
                    nxgl_coord_t radius,
                    FAR struct nxgl_point_s *circle)
{
  nxgl_coord_t xoffs;
  nxgl_coord_t yoffs;

  /* 0, 90, 180, and 270 degrees are the easiest */

  circle[POINT_0p0].x   = center->x + radius;
  circle[POINT_0p0].y   = center->y;
  circle[POINT_90p0].x  = center->x;
  circle[POINT_90p0].y  = center->y + radius;
  circle[POINT_180p0].x = center->x - radius;
  circle[POINT_180p0].y = center->y;
  circle[POINT_270p0].x = center->x;
  circle[POINT_270p0].y = center->y - radius;

  /* Now 22.5, 67.5, 112.5, 157.5, 202.5, 247.5, 292.5, and 337.5 */

  xoffs = b16toi(b16muli(COS_22p5, radius) + b16HALF);
  yoffs = b16toi(b16muli(SIN_22p5, radius) + b16HALF);

  circle[POINT_22p5].x  = center->x + xoffs;
  circle[POINT_22p5].y  = center->y + yoffs;
  circle[POINT_157p5].x = center->x - xoffs;
  circle[POINT_157p5].y = center->y + yoffs;
  circle[POINT_202p5].x = center->x - xoffs;
  circle[POINT_202p5].y = center->y - yoffs;
  circle[POINT_337p5].x = center->x + xoffs;
  circle[POINT_337p5].y = center->y - yoffs;

  circle[POINT_67p5].x  = center->x + yoffs;
  circle[POINT_67p5].y  = center->y + xoffs;
  circle[POINT_112p5].x = center->x - yoffs;
  circle[POINT_112p5].y = center->y + xoffs;
  circle[POINT_247p5].x = center->x - yoffs;
  circle[POINT_247p5].y = center->y - xoffs;
  circle[POINT_292p5].x = center->x + yoffs;
  circle[POINT_292p5].y = center->y - xoffs;

  /* Finally, 45.0, 135.0, 225.0, 315.0 */

  xoffs = b16toi(b16muli(COS_45p0, radius) + b16HALF);

  circle[POINT_45p0].x  = center->x + xoffs;
  circle[POINT_45p0].y  = center->y + xoffs;
  circle[POINT_135p0].x = center->x - xoffs;
  circle[POINT_135p0].y = center->y + xoffs;
  circle[POINT_225p0].x = center->x - xoffs;
  circle[POINT_225p0].y = center->y - xoffs;
  circle[POINT_315p0].x = center->x + xoffs;
  circle[POINT_315p0].y = center->y - xoffs;
}
