/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.core.client.impl;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.accumulo.core.Constants;
import org.apache.accumulo.core.client.AccumuloException;
import org.apache.accumulo.core.client.AccumuloSecurityException;
import org.apache.accumulo.core.client.Instance;
import org.apache.accumulo.core.client.IsolatedScanner;
import org.apache.accumulo.core.client.IteratorSetting;
import org.apache.accumulo.core.client.NamespaceExistsException;
import org.apache.accumulo.core.client.NamespaceNotFoundException;
import org.apache.accumulo.core.client.RowIterator;
import org.apache.accumulo.core.client.Scanner;
import org.apache.accumulo.core.client.TableDeletedException;
import org.apache.accumulo.core.client.TableExistsException;
import org.apache.accumulo.core.client.TableNotFoundException;
import org.apache.accumulo.core.client.TableOfflineException;
import org.apache.accumulo.core.client.admin.DiskUsage;
import org.apache.accumulo.core.client.admin.FindMax;
import org.apache.accumulo.core.client.admin.TableOperations;
import org.apache.accumulo.core.client.admin.TimeType;
import org.apache.accumulo.core.client.impl.AccumuloServerException;
import org.apache.accumulo.core.client.impl.ClientExec;
import org.apache.accumulo.core.client.impl.ClientExecReturn;
import org.apache.accumulo.core.client.impl.MasterClient;
import org.apache.accumulo.core.client.impl.ServerClient;
import org.apache.accumulo.core.client.impl.ServerConfigurationUtil;
import org.apache.accumulo.core.client.impl.TableOperationsHelper;
import org.apache.accumulo.core.client.impl.Tables;
import org.apache.accumulo.core.client.impl.TabletLocator;
import org.apache.accumulo.core.client.impl.thrift.ClientService;
import org.apache.accumulo.core.client.impl.thrift.TDiskUsage;
import org.apache.accumulo.core.client.impl.thrift.ThriftSecurityException;
import org.apache.accumulo.core.client.impl.thrift.ThriftTableOperationException;
import org.apache.accumulo.core.conf.ConfigurationCopy;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.constraints.Constraint;
import org.apache.accumulo.core.data.ByteSequence;
import org.apache.accumulo.core.data.Key;
import org.apache.accumulo.core.data.KeyExtent;
import org.apache.accumulo.core.data.Range;
import org.apache.accumulo.core.data.Value;
import org.apache.accumulo.core.iterators.IteratorUtil;
import org.apache.accumulo.core.iterators.SortedKeyValueIterator;
import org.apache.accumulo.core.master.state.tables.TableState;
import org.apache.accumulo.core.master.thrift.FateOperation;
import org.apache.accumulo.core.master.thrift.MasterClientService;
import org.apache.accumulo.core.metadata.MetadataServicer;
import org.apache.accumulo.core.metadata.schema.MetadataSchema;
import org.apache.accumulo.core.security.Authorizations;
import org.apache.accumulo.core.security.Credentials;
import org.apache.accumulo.core.tabletserver.thrift.NotServingTabletException;
import org.apache.accumulo.core.tabletserver.thrift.TabletClientService;
import org.apache.accumulo.core.util.ArgumentChecker;
import org.apache.accumulo.core.util.ByteBufferUtil;
import org.apache.accumulo.core.util.CachedConfiguration;
import org.apache.accumulo.core.util.LocalityGroupUtil;
import org.apache.accumulo.core.util.MapCounter;
import org.apache.accumulo.core.util.NamingThreadFactory;
import org.apache.accumulo.core.util.OpTimer;
import org.apache.accumulo.core.util.Pair;
import org.apache.accumulo.core.util.StringUtil;
import org.apache.accumulo.core.util.TextUtil;
import org.apache.accumulo.core.util.ThriftUtil;
import org.apache.accumulo.core.util.UtilWaitThread;
import org.apache.accumulo.core.volume.VolumeConfiguration;
import org.apache.accumulo.trace.instrument.Tracer;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.Text;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.thrift.TApplicationException;
import org.apache.thrift.TException;
import org.apache.thrift.transport.TTransportException;

public class TableOperationsImpl
extends TableOperationsHelper {
    private Instance instance;
    private Credentials credentials;
    public static final String CLONE_EXCLUDE_PREFIX = "!";
    private static final Logger log = Logger.getLogger(TableOperations.class);

    public TableOperationsImpl(Instance instance, Credentials credentials) {
        ArgumentChecker.notNull(instance, credentials);
        this.instance = instance;
        this.credentials = credentials;
    }

    @Override
    public SortedSet<String> list() {
        OpTimer opTimer = new OpTimer(log, Level.TRACE).start("Fetching list of tables...");
        TreeSet<String> tableNames = new TreeSet<String>(Tables.getNameToIdMap(this.instance).keySet());
        opTimer.stop("Fetched " + tableNames.size() + " table names in %DURATION%");
        return tableNames;
    }

    @Override
    public boolean exists(String tableName) {
        ArgumentChecker.notNull(tableName);
        if (tableName.equals("accumulo.metadata") || tableName.equals("accumulo.root")) {
            return true;
        }
        OpTimer opTimer = new OpTimer(log, Level.TRACE).start("Checking if table " + tableName + " exists...");
        boolean exists = Tables.getNameToIdMap(this.instance).containsKey(tableName);
        opTimer.stop("Checked existance of " + exists + " in %DURATION%");
        return exists;
    }

    @Override
    public void create(String tableName) throws AccumuloException, AccumuloSecurityException, TableExistsException {
        this.create(tableName, true, TimeType.MILLIS);
    }

    @Override
    public void create(String tableName, boolean limitVersion) throws AccumuloException, AccumuloSecurityException, TableExistsException {
        this.create(tableName, limitVersion, TimeType.MILLIS);
    }

    @Override
    public void create(String tableName, boolean limitVersion, TimeType timeType) throws AccumuloException, AccumuloSecurityException, TableExistsException {
        ArgumentChecker.notNull(tableName, (Object)timeType);
        List<ByteBuffer> args = Arrays.asList(ByteBuffer.wrap(tableName.getBytes(Constants.UTF8)), ByteBuffer.wrap(timeType.name().getBytes(Constants.UTF8)));
        Map<String, String> opts = limitVersion ? IteratorUtil.generateInitialTableProperties(limitVersion) : Collections.emptyMap();
        try {
            this.doTableFateOperation(tableName, AccumuloException.class, FateOperation.TABLE_CREATE, args, opts);
        }
        catch (TableNotFoundException e) {
            throw new AssertionError((Object)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long beginFateOperation() throws ThriftSecurityException, TException {
        while (true) {
            MasterClientService.Client client = null;
            try {
                client = MasterClient.getConnectionWithRetry(this.instance);
                long l = client.beginFateOperation(Tracer.traceInfo(), this.credentials.toThrift(this.instance));
                return l;
            }
            catch (TTransportException tte) {
                log.debug((Object)"Failed to call beginFateOperation(), retrying ... ", (Throwable)tte);
                UtilWaitThread.sleep(100L);
                continue;
            }
            finally {
                MasterClient.close(client);
                continue;
            }
            break;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void executeFateOperation(long opid, FateOperation op, List<ByteBuffer> args, Map<String, String> opts, boolean autoCleanUp) throws ThriftSecurityException, TException, ThriftTableOperationException {
        while (true) {
            MasterClientService.Client client = null;
            try {
                client = MasterClient.getConnectionWithRetry(this.instance);
                client.executeFateOperation(Tracer.traceInfo(), this.credentials.toThrift(this.instance), opid, op, args, opts, autoCleanUp);
            }
            catch (TTransportException tte) {
                log.debug((Object)"Failed to call executeFateOperation(), retrying ... ", (Throwable)tte);
                UtilWaitThread.sleep(100L);
                continue;
            }
            finally {
                MasterClient.close(client);
                continue;
            }
            break;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String waitForFateOperation(long opid) throws ThriftSecurityException, TException, ThriftTableOperationException {
        while (true) {
            MasterClientService.Client client = null;
            try {
                client = MasterClient.getConnectionWithRetry(this.instance);
                String string = client.waitForFateOperation(Tracer.traceInfo(), this.credentials.toThrift(this.instance), opid);
                return string;
            }
            catch (TTransportException tte) {
                log.debug((Object)"Failed to call waitForFateOperation(), retrying ... ", (Throwable)tte);
                UtilWaitThread.sleep(100L);
                continue;
            }
            finally {
                MasterClient.close(client);
                continue;
            }
            break;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void finishFateOperation(long opid) throws ThriftSecurityException, TException {
        while (true) {
            MasterClientService.Client client = null;
            try {
                client = MasterClient.getConnectionWithRetry(this.instance);
                client.finishFateOperation(Tracer.traceInfo(), this.credentials.toThrift(this.instance), opid);
            }
            catch (TTransportException tte) {
                log.debug((Object)"Failed to call finishFateOperation(), retrying ... ", (Throwable)tte);
                UtilWaitThread.sleep(100L);
                continue;
            }
            finally {
                MasterClient.close(client);
                continue;
            }
            break;
        }
    }

    String doFateOperation(FateOperation op, List<ByteBuffer> args, Map<String, String> opts) throws AccumuloSecurityException, TableExistsException, TableNotFoundException, AccumuloException, NamespaceExistsException, NamespaceNotFoundException {
        return this.doFateOperation(op, args, opts, true);
    }

    String doFateOperation(FateOperation op, List<ByteBuffer> args, Map<String, String> opts, boolean wait) throws AccumuloSecurityException, TableExistsException, TableNotFoundException, AccumuloException, NamespaceExistsException, NamespaceNotFoundException {
        Long opid = null;
        try {
            String ret;
            opid = this.beginFateOperation();
            this.executeFateOperation(opid, op, args, opts, !wait);
            if (!wait) {
                opid = null;
                String string = null;
                return string;
            }
            String e = ret = this.waitForFateOperation(opid);
            return e;
        }
        catch (ThriftSecurityException e) {
            String tableName = ByteBufferUtil.toString(args.get(0));
            switch (e.getCode()) {
                case TABLE_DOESNT_EXIST: {
                    throw new TableNotFoundException(null, tableName, "Target table does not exist");
                }
                case NAMESPACE_DOESNT_EXIST: {
                    throw new NamespaceNotFoundException(null, tableName, "Target namespace does not exist");
                }
            }
            String tableInfo = Tables.getPrintableTableInfoFromName(this.instance, tableName);
            throw new AccumuloSecurityException(e.user, e.code, tableInfo, (Throwable)((Object)e));
        }
        catch (ThriftTableOperationException e) {
            switch (e.getType()) {
                case EXISTS: {
                    throw new TableExistsException(e);
                }
                case NOTFOUND: {
                    throw new TableNotFoundException(e);
                }
                case NAMESPACE_EXISTS: {
                    throw new NamespaceExistsException(e);
                }
                case NAMESPACE_NOTFOUND: {
                    throw new NamespaceNotFoundException(e);
                }
                case OFFLINE: {
                    throw new TableOfflineException(this.instance, null);
                }
            }
            throw new AccumuloException(e.description, (Throwable)((Object)e));
        }
        catch (Exception e) {
            throw new AccumuloException(e.getMessage(), e);
        }
        finally {
            Tables.clearCache(this.instance);
            if (opid != null) {
                try {
                    this.finishFateOperation(opid);
                }
                catch (Exception e) {
                    log.warn((Object)e.getMessage(), (Throwable)e);
                }
            }
        }
    }

    @Override
    public void addSplits(String tableName, SortedSet<Text> partitionKeys) throws TableNotFoundException, AccumuloException, AccumuloSecurityException {
        String tableId = Tables.getTableId(this.instance, tableName);
        ArrayList<Text> splits = new ArrayList<Text>(partitionKeys);
        Collections.sort(splits);
        CountDownLatch latch = new CountDownLatch(splits.size());
        AtomicReference<Object> exception = new AtomicReference<Object>(null);
        ExecutorService executor = Executors.newFixedThreadPool(16, new NamingThreadFactory("addSplits"));
        try {
            executor.submit(new SplitTask(new SplitEnv(tableName, tableId, executor, latch, exception), splits));
            while (!latch.await(100L, TimeUnit.MILLISECONDS)) {
                if (exception.get() == null) continue;
                executor.shutdownNow();
                Exception excep = exception.get();
                if (excep instanceof TableNotFoundException) {
                    throw (TableNotFoundException)excep;
                }
                if (excep instanceof AccumuloException) {
                    throw (AccumuloException)excep;
                }
                if (excep instanceof AccumuloSecurityException) {
                    throw (AccumuloSecurityException)excep;
                }
                if (excep instanceof RuntimeException) {
                    throw (RuntimeException)excep;
                }
                throw new RuntimeException(excep);
            }
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        finally {
            executor.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addSplits(String tableName, SortedSet<Text> partitionKeys, String tableId) throws AccumuloException, AccumuloSecurityException, TableNotFoundException, AccumuloServerException {
        TabletLocator tabLocator = TabletLocator.getLocator(this.instance, new Text(tableId));
        for (Text split : partitionKeys) {
            boolean successful = false;
            int attempt = 0;
            while (!successful) {
                block16: {
                    if (attempt > 0) {
                        UtilWaitThread.sleep(100L);
                    }
                    ++attempt;
                    TabletLocator.TabletLocation tl = tabLocator.locateTablet(this.credentials, split, false, false);
                    if (tl == null) {
                        if (!Tables.exists(this.instance, tableId)) {
                            throw new TableNotFoundException(tableId, tableName, null);
                        }
                        if (Tables.getTableState(this.instance, tableId) != TableState.OFFLINE) continue;
                        throw new TableOfflineException(this.instance, tableId);
                    }
                    try {
                        TabletClientService.Client client = ThriftUtil.getTServerClient(tl.tablet_location, ServerConfigurationUtil.getConfiguration(this.instance));
                        try {
                            OpTimer opTimer = null;
                            if (log.isTraceEnabled()) {
                                opTimer = new OpTimer(log, Level.TRACE).start("Splitting tablet " + tl.tablet_extent + " on " + tl.tablet_location + " at " + split);
                            }
                            client.splitTablet(Tracer.traceInfo(), this.credentials.toThrift(this.instance), tl.tablet_extent.toThrift(), TextUtil.getByteBuffer(split));
                            tabLocator.invalidateCache(tl.tablet_extent);
                            if (opTimer == null) break block16;
                            opTimer.stop("Split tablet in %DURATION%");
                        }
                        finally {
                            ThriftUtil.returnClient(client);
                        }
                    }
                    catch (TApplicationException tae) {
                        throw new AccumuloServerException(tl.tablet_location, tae);
                    }
                    catch (TTransportException e) {
                        tabLocator.invalidateCache(tl.tablet_location);
                        continue;
                    }
                    catch (ThriftSecurityException e) {
                        Tables.clearCache(this.instance);
                        if (!Tables.exists(this.instance, tableId)) {
                            throw new TableNotFoundException(tableId, tableName, null);
                        }
                        throw new AccumuloSecurityException(e.user, e.code, (Throwable)((Object)e));
                    }
                    catch (NotServingTabletException e) {
                        tabLocator.invalidateCache(tl.tablet_extent);
                        continue;
                    }
                    catch (TException e) {
                        tabLocator.invalidateCache(tl.tablet_location);
                        continue;
                    }
                }
                successful = true;
            }
        }
    }

    @Override
    public void merge(String tableName, Text start, Text end) throws AccumuloException, AccumuloSecurityException, TableNotFoundException {
        ArgumentChecker.notNull(tableName);
        ByteBuffer EMPTY = ByteBuffer.allocate(0);
        List<ByteBuffer> args = Arrays.asList(ByteBuffer.wrap(tableName.getBytes(Constants.UTF8)), start == null ? EMPTY : TextUtil.getByteBuffer(start), end == null ? EMPTY : TextUtil.getByteBuffer(end));
        HashMap<String, String> opts = new HashMap<String, String>();
        try {
            this.doTableFateOperation(tableName, TableNotFoundException.class, FateOperation.TABLE_MERGE, args, opts);
        }
        catch (TableExistsException e) {
            throw new AssertionError((Object)e);
        }
    }

    @Override
    public void deleteRows(String tableName, Text start, Text end) throws AccumuloException, AccumuloSecurityException, TableNotFoundException {
        ArgumentChecker.notNull(tableName);
        ByteBuffer EMPTY = ByteBuffer.allocate(0);
        List<ByteBuffer> args = Arrays.asList(ByteBuffer.wrap(tableName.getBytes(Constants.UTF8)), start == null ? EMPTY : TextUtil.getByteBuffer(start), end == null ? EMPTY : TextUtil.getByteBuffer(end));
        HashMap<String, String> opts = new HashMap<String, String>();
        try {
            this.doTableFateOperation(tableName, TableNotFoundException.class, FateOperation.TABLE_DELETE_RANGE, args, opts);
        }
        catch (TableExistsException e) {
            throw new AssertionError((Object)e);
        }
    }

    @Override
    public Collection<Text> listSplits(String tableName) throws TableNotFoundException, AccumuloSecurityException {
        ArgumentChecker.notNull(tableName);
        String tableId = Tables.getTableId(this.instance, tableName);
        TreeMap<KeyExtent, String> tabletLocations = new TreeMap<KeyExtent, String>();
        while (true) {
            try {
                tabletLocations.clear();
                MetadataServicer.forTableId(this.instance, this.credentials, tableId).getTabletLocations(tabletLocations);
            }
            catch (AccumuloSecurityException ase) {
                throw ase;
            }
            catch (Exception e) {
                if (!Tables.exists(this.instance, tableId)) {
                    throw new TableNotFoundException(tableId, tableName, null);
                }
                if (e instanceof RuntimeException && e.getCause() instanceof AccumuloSecurityException) {
                    throw (AccumuloSecurityException)e.getCause();
                }
                log.info((Object)(e.getMessage() + " ... retrying ..."));
                UtilWaitThread.sleep(3000L);
                continue;
            }
            break;
        }
        ArrayList<Text> endRows = new ArrayList<Text>(tabletLocations.size());
        for (KeyExtent ke : tabletLocations.keySet()) {
            if (ke.getEndRow() == null) continue;
            endRows.add(ke.getEndRow());
        }
        return endRows;
    }

    @Override
    @Deprecated
    public Collection<Text> getSplits(String tableName) throws TableNotFoundException {
        try {
            return this.listSplits(tableName);
        }
        catch (AccumuloSecurityException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Collection<Text> listSplits(String tableName, int maxSplits) throws TableNotFoundException, AccumuloSecurityException {
        Collection<Text> endRows = this.listSplits(tableName);
        if (endRows.size() <= maxSplits) {
            return endRows;
        }
        double r = (double)(maxSplits + 1) / (double)endRows.size();
        double pos = 0.0;
        ArrayList<Text> subset = new ArrayList<Text>(maxSplits);
        int j = 0;
        for (int i = 0; i < endRows.size() && j < maxSplits; ++i) {
            pos += r;
            while (pos > 1.0) {
                subset.add((Text)((ArrayList)endRows).get(i));
                ++j;
                pos -= 1.0;
            }
        }
        return subset;
    }

    @Override
    @Deprecated
    public Collection<Text> getSplits(String tableName, int maxSplits) throws TableNotFoundException {
        try {
            return this.listSplits(tableName, maxSplits);
        }
        catch (AccumuloSecurityException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void delete(String tableName) throws AccumuloException, AccumuloSecurityException, TableNotFoundException {
        ArgumentChecker.notNull(tableName);
        List<ByteBuffer> args = Arrays.asList(ByteBuffer.wrap(tableName.getBytes(Constants.UTF8)));
        HashMap<String, String> opts = new HashMap<String, String>();
        try {
            this.doTableFateOperation(tableName, TableNotFoundException.class, FateOperation.TABLE_DELETE, args, opts);
        }
        catch (TableExistsException e) {
            throw new AssertionError((Object)e);
        }
    }

    @Override
    public void clone(String srcTableName, String newTableName, boolean flush2, Map<String, String> propertiesToSet, Set<String> propertiesToExclude) throws AccumuloSecurityException, TableNotFoundException, AccumuloException, TableExistsException {
        ArgumentChecker.notNull(srcTableName, newTableName);
        String srcTableId = Tables.getTableId(this.instance, srcTableName);
        if (flush2) {
            this._flush(srcTableId, null, null, true);
        }
        if (propertiesToExclude == null) {
            propertiesToExclude = Collections.emptySet();
        }
        if (propertiesToSet == null) {
            propertiesToSet = Collections.emptyMap();
        }
        List<ByteBuffer> args = Arrays.asList(ByteBuffer.wrap(srcTableId.getBytes(Constants.UTF8)), ByteBuffer.wrap(newTableName.getBytes(Constants.UTF8)));
        HashMap<String, String> opts = new HashMap<String, String>();
        for (Map.Entry<String, String> entry : propertiesToSet.entrySet()) {
            if (entry.getKey().startsWith(CLONE_EXCLUDE_PREFIX)) {
                throw new IllegalArgumentException("Property can not start with !");
            }
            opts.put(entry.getKey(), entry.getValue());
        }
        for (String prop : propertiesToExclude) {
            opts.put(CLONE_EXCLUDE_PREFIX + prop, "");
        }
        this.doTableFateOperation(newTableName, AccumuloException.class, FateOperation.TABLE_CLONE, args, opts);
    }

    @Override
    public void rename(String oldTableName, String newTableName) throws AccumuloSecurityException, TableNotFoundException, AccumuloException, TableExistsException {
        List<ByteBuffer> args = Arrays.asList(ByteBuffer.wrap(oldTableName.getBytes(Constants.UTF8)), ByteBuffer.wrap(newTableName.getBytes(Constants.UTF8)));
        HashMap<String, String> opts = new HashMap<String, String>();
        this.doTableFateOperation(oldTableName, TableNotFoundException.class, FateOperation.TABLE_RENAME, args, opts);
    }

    @Override
    @Deprecated
    public void flush(String tableName) throws AccumuloException, AccumuloSecurityException {
        try {
            this.flush(tableName, null, null, false);
        }
        catch (TableNotFoundException e) {
            throw new AccumuloException(e.getMessage(), e);
        }
    }

    @Override
    public void flush(String tableName, Text start, Text end, boolean wait) throws AccumuloException, AccumuloSecurityException, TableNotFoundException {
        ArgumentChecker.notNull(tableName);
        String tableId = Tables.getTableId(this.instance, tableName);
        this._flush(tableId, start, end, wait);
    }

    @Override
    public void compact(String tableName, Text start, Text end, boolean flush2, boolean wait) throws AccumuloSecurityException, TableNotFoundException, AccumuloException {
        this.compact(tableName, start, end, new ArrayList<IteratorSetting>(), flush2, wait);
    }

    @Override
    public void compact(String tableName, Text start, Text end, List<IteratorSetting> iterators, boolean flush2, boolean wait) throws AccumuloSecurityException, TableNotFoundException, AccumuloException {
        ArgumentChecker.notNull(tableName);
        ByteBuffer EMPTY = ByteBuffer.allocate(0);
        String tableId = Tables.getTableId(this.instance, tableName);
        if (flush2) {
            this._flush(tableId, start, end, true);
        }
        List<ByteBuffer> args = Arrays.asList(ByteBuffer.wrap(tableId.getBytes(Constants.UTF8)), start == null ? EMPTY : TextUtil.getByteBuffer(start), end == null ? EMPTY : TextUtil.getByteBuffer(end), ByteBuffer.wrap(IteratorUtil.encodeIteratorSettings(iterators)));
        HashMap<String, String> opts = new HashMap<String, String>();
        try {
            this.doFateOperation(FateOperation.TABLE_COMPACT, args, opts, wait);
        }
        catch (TableExistsException e) {
            throw new AssertionError((Object)e);
        }
        catch (NamespaceExistsException e) {
            throw new AssertionError((Object)e);
        }
        catch (NamespaceNotFoundException e) {
            throw new TableNotFoundException(null, tableName, "Namespace not found", e);
        }
    }

    @Override
    public void cancelCompaction(String tableName) throws AccumuloSecurityException, TableNotFoundException, AccumuloException {
        String tableId = Tables.getTableId(this.instance, tableName);
        List<ByteBuffer> args = Arrays.asList(ByteBuffer.wrap(tableId.getBytes(Constants.UTF8)));
        HashMap<String, String> opts = new HashMap<String, String>();
        try {
            this.doTableFateOperation(tableName, TableNotFoundException.class, FateOperation.TABLE_CANCEL_COMPACT, args, opts);
        }
        catch (TableExistsException e) {
            throw new AssertionError((Object)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void _flush(String tableId, Text start, Text end, boolean wait) throws AccumuloException, AccumuloSecurityException, TableNotFoundException {
        try {
            long flushID;
            MasterClientService.Client client;
            while (true) {
                client = null;
                try {
                    client = MasterClient.getConnectionWithRetry(this.instance);
                    flushID = client.initiateFlush(Tracer.traceInfo(), this.credentials.toThrift(this.instance), tableId);
                }
                catch (TTransportException tte) {
                    log.debug((Object)"Failed to call initiateFlush, retrying ... ", (Throwable)tte);
                    UtilWaitThread.sleep(100L);
                    continue;
                }
                finally {
                    MasterClient.close(client);
                    continue;
                }
                break;
            }
            while (true) {
                client = null;
                try {
                    client = MasterClient.getConnectionWithRetry(this.instance);
                    client.waitForFlush(Tracer.traceInfo(), this.credentials.toThrift(this.instance), tableId, TextUtil.getByteBuffer(start), TextUtil.getByteBuffer(end), flushID, wait ? Long.MAX_VALUE : 1L);
                }
                catch (TTransportException tte) {
                    log.debug((Object)"Failed to call initiateFlush, retrying ... ", (Throwable)tte);
                    UtilWaitThread.sleep(100L);
                    continue;
                }
                finally {
                    MasterClient.close(client);
                    continue;
                }
                break;
            }
        }
        catch (ThriftSecurityException e) {
            switch (e.getCode()) {
                case TABLE_DOESNT_EXIST: {
                    throw new TableNotFoundException(tableId, null, e.getMessage(), (Throwable)((Object)e));
                }
            }
            log.debug((Object)("flush security exception on table id " + tableId));
            throw new AccumuloSecurityException(e.user, e.code, (Throwable)((Object)e));
        }
        catch (ThriftTableOperationException e) {
            switch (e.getType()) {
                case NOTFOUND: {
                    throw new TableNotFoundException(e);
                }
            }
            throw new AccumuloException(e.description, (Throwable)((Object)e));
        }
        catch (Exception e) {
            throw new AccumuloException(e);
        }
    }

    @Override
    public void setProperty(final String tableName, final String property, final String value) throws AccumuloException, AccumuloSecurityException {
        ArgumentChecker.notNull(tableName, property, value);
        try {
            MasterClient.executeTable(this.instance, new ClientExec<MasterClientService.Client>(){

                @Override
                public void execute(MasterClientService.Client client) throws Exception {
                    client.setTableProperty(Tracer.traceInfo(), TableOperationsImpl.this.credentials.toThrift(TableOperationsImpl.this.instance), tableName, property, value);
                }
            });
        }
        catch (TableNotFoundException e) {
            throw new AccumuloException(e);
        }
    }

    @Override
    public void removeProperty(final String tableName, final String property) throws AccumuloException, AccumuloSecurityException {
        ArgumentChecker.notNull(tableName, property);
        try {
            MasterClient.executeTable(this.instance, new ClientExec<MasterClientService.Client>(){

                @Override
                public void execute(MasterClientService.Client client) throws Exception {
                    client.removeTableProperty(Tracer.traceInfo(), TableOperationsImpl.this.credentials.toThrift(TableOperationsImpl.this.instance), tableName, property);
                }
            });
        }
        catch (TableNotFoundException e) {
            throw new AccumuloException(e);
        }
    }

    @Override
    public Iterable<Map.Entry<String, String>> getProperties(final String tableName) throws AccumuloException, TableNotFoundException {
        ArgumentChecker.notNull(tableName);
        try {
            return ServerClient.executeRaw(this.instance, new ClientExecReturn<Map<String, String>, ClientService.Client>(){

                @Override
                public Map<String, String> execute(ClientService.Client client) throws Exception {
                    return client.getTableConfiguration(Tracer.traceInfo(), TableOperationsImpl.this.credentials.toThrift(TableOperationsImpl.this.instance), tableName);
                }
            }).entrySet();
        }
        catch (ThriftTableOperationException e) {
            switch (e.getType()) {
                case NOTFOUND: {
                    throw new TableNotFoundException(e);
                }
                case NAMESPACE_NOTFOUND: {
                    throw new TableNotFoundException(tableName, new NamespaceNotFoundException(e));
                }
            }
            throw new AccumuloException(e.description, (Throwable)((Object)e));
        }
        catch (AccumuloException e) {
            throw e;
        }
        catch (Exception e) {
            throw new AccumuloException(e);
        }
    }

    @Override
    public void setLocalityGroups(String tableName, Map<String, Set<Text>> groups) throws AccumuloException, AccumuloSecurityException, TableNotFoundException {
        HashSet all = new HashSet();
        for (Map.Entry<String, Set<Text>> entry : groups.entrySet()) {
            if (!Collections.disjoint(all, (Collection)entry.getValue())) {
                throw new IllegalArgumentException("Group " + entry.getKey() + " overlaps with another group");
            }
            all.addAll(entry.getValue());
        }
        for (Map.Entry<String, Set<Text>> entry : groups.entrySet()) {
            Set<Text> colFams = entry.getValue();
            String value = LocalityGroupUtil.encodeColumnFamilies(colFams);
            this.setProperty(tableName, (Object)((Object)Property.TABLE_LOCALITY_GROUP_PREFIX) + entry.getKey(), value);
        }
        try {
            this.setProperty(tableName, Property.TABLE_LOCALITY_GROUPS.getKey(), StringUtil.join(groups.keySet(), ","));
        }
        catch (AccumuloException e) {
            if (e.getCause() instanceof TableNotFoundException) {
                throw (TableNotFoundException)e.getCause();
            }
            throw e;
        }
        String prefix = Property.TABLE_LOCALITY_GROUP_PREFIX.getKey();
        for (Map.Entry<String, String> entry : this.getProperties(tableName)) {
            String[] parts;
            String group;
            String property = entry.getKey();
            if (!property.startsWith(prefix) || groups.containsKey(group = (parts = property.split("\\."))[parts.length - 1])) continue;
            this.removeProperty(tableName, property);
        }
    }

    @Override
    public Map<String, Set<Text>> getLocalityGroups(String tableName) throws AccumuloException, TableNotFoundException {
        ConfigurationCopy conf = new ConfigurationCopy(this.getProperties(tableName));
        Map<String, Set<ByteSequence>> groups = LocalityGroupUtil.getLocalityGroups(conf);
        HashMap<String, Set<Text>> groups2 = new HashMap<String, Set<Text>>();
        for (Map.Entry<String, Set<ByteSequence>> entry : groups.entrySet()) {
            HashSet<Text> colFams = new HashSet<Text>();
            for (ByteSequence bs : entry.getValue()) {
                colFams.add(new Text(bs.toArray()));
            }
            groups2.put(entry.getKey(), colFams);
        }
        return groups2;
    }

    @Override
    public Set<Range> splitRangeByTablets(String tableName, Range range, int maxSplits) throws AccumuloException, AccumuloSecurityException, TableNotFoundException {
        ArgumentChecker.notNull(tableName, range);
        if (maxSplits < 1) {
            throw new IllegalArgumentException("maximum splits must be >= 1");
        }
        if (maxSplits == 1) {
            return Collections.singleton(range);
        }
        Random random = new Random();
        HashMap<String, Map<KeyExtent, List<Range>>> binnedRanges = new HashMap<String, Map<KeyExtent, List<Range>>>();
        String tableId = Tables.getTableId(this.instance, tableName);
        TabletLocator tl = TabletLocator.getLocator(this.instance, new Text(tableId));
        tl.invalidateCache();
        while (!tl.binRanges(this.credentials, Collections.singletonList(range), binnedRanges).isEmpty()) {
            if (!Tables.exists(this.instance, tableId)) {
                throw new TableDeletedException(tableId);
            }
            if (Tables.getTableState(this.instance, tableId) == TableState.OFFLINE) {
                throw new TableOfflineException(this.instance, tableId);
            }
            log.warn((Object)"Unable to locate bins for specified range. Retrying.");
            UtilWaitThread.sleep(100 + random.nextInt(100));
            binnedRanges.clear();
            tl.invalidateCache();
        }
        LinkedList<Object> unmergedExtents = new LinkedList<Object>();
        ArrayList<Object> mergedExtents = new ArrayList<Object>();
        for (Map map : binnedRanges.values()) {
            unmergedExtents.addAll(map.keySet());
        }
        Collections.sort(unmergedExtents);
        while (unmergedExtents.size() + mergedExtents.size() > maxSplits) {
            if (unmergedExtents.size() >= 2) {
                KeyExtent first = (KeyExtent)unmergedExtents.removeFirst();
                KeyExtent second = (KeyExtent)unmergedExtents.removeFirst();
                first.setEndRow(second.getEndRow());
                mergedExtents.add(first);
                continue;
            }
            mergedExtents.addAll(unmergedExtents);
            unmergedExtents.clear();
            unmergedExtents.addAll(mergedExtents);
            mergedExtents.clear();
        }
        mergedExtents.addAll(unmergedExtents);
        HashSet<Range> ranges = new HashSet<Range>();
        for (KeyExtent keyExtent : mergedExtents) {
            ranges.add(keyExtent.toDataRange().clip(range));
        }
        return ranges;
    }

    private Path checkPath(String dir, String kind, String type) throws IOException, AccumuloException {
        FileStatus[] listStatus;
        Path ret;
        FileSystem fs = VolumeConfiguration.getVolume(dir, CachedConfiguration.getInstance(), ServerConfigurationUtil.getConfiguration(this.instance)).getFileSystem();
        if (!fs.exists(ret = dir.contains(":") ? new Path(dir) : fs.makeQualified(new Path(dir)))) {
            throw new AccumuloException(kind + " import " + type + " directory " + dir + " does not exist!");
        }
        if (!fs.getFileStatus(ret).isDir()) {
            throw new AccumuloException(kind + " import " + type + " directory " + dir + " is not a directory!");
        }
        if (type.equals("failure") && (listStatus = fs.listStatus(ret)) != null && listStatus.length != 0) {
            throw new AccumuloException("Bulk import failure directory " + ret + " is not empty");
        }
        return ret;
    }

    @Override
    public void importDirectory(String tableName, String dir, String failureDir, boolean setTime) throws IOException, AccumuloSecurityException, TableNotFoundException, AccumuloException {
        ArgumentChecker.notNull(tableName, dir, failureDir);
        Tables.getTableId(this.instance, tableName);
        Path dirPath = this.checkPath(dir, "Bulk", "");
        Path failPath = this.checkPath(failureDir, "Bulk", "failure");
        List<ByteBuffer> args = Arrays.asList(ByteBuffer.wrap(tableName.getBytes(Constants.UTF8)), ByteBuffer.wrap(dirPath.toString().getBytes(Constants.UTF8)), ByteBuffer.wrap(failPath.toString().getBytes(Constants.UTF8)), ByteBuffer.wrap((setTime + "").getBytes(Constants.UTF8)));
        HashMap<String, String> opts = new HashMap<String, String>();
        try {
            this.doTableFateOperation(tableName, TableNotFoundException.class, FateOperation.TABLE_BULK_IMPORT, args, opts);
        }
        catch (TableExistsException e) {
            throw new AssertionError((Object)e);
        }
    }

    private void waitForTableStateTransition(String tableId, TableState expectedState) throws AccumuloException, TableNotFoundException, AccumuloSecurityException {
        Text startRow = null;
        Text lastRow = null;
        while (true) {
            long waitTime;
            if (Tables.getTableState(this.instance, tableId) != expectedState) {
                Tables.clearCache(this.instance);
                if (Tables.getTableState(this.instance, tableId) != expectedState) {
                    if (!Tables.exists(this.instance, tableId)) {
                        throw new TableDeletedException(tableId);
                    }
                    throw new AccumuloException("Unexpected table state " + tableId + " " + (Object)((Object)Tables.getTableState(this.instance, tableId)) + " != " + (Object)((Object)expectedState));
                }
            }
            Range range = new KeyExtent(new Text(tableId), null, null).toMetadataRange();
            range = startRow == null || lastRow == null ? new KeyExtent(new Text(tableId), null, null).toMetadataRange() : new Range(startRow, lastRow);
            String metaTable = "accumulo.metadata";
            if (tableId.equals("!0")) {
                metaTable = "accumulo.root";
            }
            Scanner scanner = this.instance.getConnector(this.credentials.getPrincipal(), this.credentials.getToken()).createScanner(metaTable, Authorizations.EMPTY);
            scanner = new IsolatedScanner(scanner);
            MetadataSchema.TabletsSection.TabletColumnFamily.PREV_ROW_COLUMN.fetch(scanner);
            scanner.fetchColumnFamily(MetadataSchema.TabletsSection.CurrentLocationColumnFamily.NAME);
            scanner.setRange(range);
            RowIterator rowIter = new RowIterator(scanner);
            KeyExtent lastExtent = null;
            int total = 0;
            int waitFor = 0;
            int holes = 0;
            Text continueRow = null;
            MapCounter<String> serverCounts = new MapCounter<String>();
            while (rowIter.hasNext()) {
                Object row = rowIter.next();
                ++total;
                KeyExtent extent = null;
                String future = null;
                String current = null;
                while (row.hasNext()) {
                    Map.Entry entry = (Map.Entry)row.next();
                    Key key = (Key)entry.getKey();
                    if (key.getColumnFamily().equals((Object)MetadataSchema.TabletsSection.FutureLocationColumnFamily.NAME)) {
                        future = ((Value)entry.getValue()).toString();
                    }
                    if (key.getColumnFamily().equals((Object)MetadataSchema.TabletsSection.CurrentLocationColumnFamily.NAME)) {
                        current = ((Value)entry.getValue()).toString();
                    }
                    if (!MetadataSchema.TabletsSection.TabletColumnFamily.PREV_ROW_COLUMN.hasColumns(key)) continue;
                    extent = new KeyExtent(key.getRow(), (Value)entry.getValue());
                }
                if (expectedState == TableState.ONLINE && current == null || expectedState == TableState.OFFLINE && (future != null || current != null)) {
                    if (continueRow == null) {
                        continueRow = extent.getMetadataEntry();
                    }
                    ++waitFor;
                    lastRow = extent.getMetadataEntry();
                    if (current != null) {
                        serverCounts.increment(current, 1L);
                    }
                    if (future != null) {
                        serverCounts.increment(future, 1L);
                    }
                }
                if (!extent.getTableId().toString().equals(tableId)) {
                    throw new AccumuloException("Saw unexpected table Id " + tableId + " " + extent);
                }
                if (lastExtent != null && !extent.isPreviousExtent(lastExtent)) {
                    ++holes;
                }
                lastExtent = extent;
            }
            if (continueRow != null) {
                startRow = continueRow;
            }
            if (holes > 0 || total == 0) {
                startRow = null;
                lastRow = null;
            }
            if (waitFor <= 0 && holes <= 0 && total != 0) break;
            long maxPerServer = 0L;
            if (serverCounts.size() > 0) {
                maxPerServer = Collections.max(serverCounts.values());
                waitTime = maxPerServer * 10L;
            } else {
                waitTime = waitFor * 10;
            }
            waitTime = Math.max(100L, waitTime);
            waitTime = Math.min(5000L, waitTime);
            log.trace((Object)("Waiting for " + waitFor + "(" + maxPerServer + ") tablets, startRow = " + startRow + " lastRow = " + lastRow + ", holes=" + holes + " sleeping:" + waitTime + "ms"));
            UtilWaitThread.sleep(waitTime);
        }
    }

    @Override
    public void offline(String tableName) throws AccumuloSecurityException, AccumuloException, TableNotFoundException {
        this.offline(tableName, false);
    }

    @Override
    public void offline(String tableName, boolean wait) throws AccumuloSecurityException, AccumuloException, TableNotFoundException {
        ArgumentChecker.notNull(tableName);
        String tableId = Tables.getTableId(this.instance, tableName);
        List<ByteBuffer> args = Arrays.asList(ByteBuffer.wrap(tableId.getBytes(Constants.UTF8)));
        HashMap<String, String> opts = new HashMap<String, String>();
        try {
            this.doTableFateOperation(tableName, TableNotFoundException.class, FateOperation.TABLE_OFFLINE, args, opts);
        }
        catch (TableExistsException e) {
            throw new AssertionError((Object)e);
        }
        if (wait) {
            this.waitForTableStateTransition(tableId, TableState.OFFLINE);
        }
    }

    @Override
    public void online(String tableName) throws AccumuloSecurityException, AccumuloException, TableNotFoundException {
        this.online(tableName, false);
    }

    @Override
    public void online(String tableName, boolean wait) throws AccumuloSecurityException, AccumuloException, TableNotFoundException {
        ArgumentChecker.notNull(tableName);
        String tableId = Tables.getTableId(this.instance, tableName);
        List<ByteBuffer> args = Arrays.asList(ByteBuffer.wrap(tableId.getBytes(Constants.UTF8)));
        HashMap<String, String> opts = new HashMap<String, String>();
        try {
            this.doTableFateOperation(tableName, TableNotFoundException.class, FateOperation.TABLE_ONLINE, args, opts);
        }
        catch (TableExistsException e) {
            throw new AssertionError((Object)e);
        }
        if (wait) {
            this.waitForTableStateTransition(tableId, TableState.ONLINE);
        }
    }

    @Override
    public void clearLocatorCache(String tableName) throws TableNotFoundException {
        ArgumentChecker.notNull(tableName);
        TabletLocator tabLocator = TabletLocator.getLocator(this.instance, new Text(Tables.getTableId(this.instance, tableName)));
        tabLocator.invalidateCache();
    }

    @Override
    public Map<String, String> tableIdMap() {
        return Tables.getNameToIdMap(this.instance);
    }

    @Override
    public Text getMaxRow(String tableName, Authorizations auths, Text startRow, boolean startInclusive, Text endRow, boolean endInclusive) throws TableNotFoundException, AccumuloException, AccumuloSecurityException {
        ArgumentChecker.notNull(tableName, auths);
        Scanner scanner = this.instance.getConnector(this.credentials.getPrincipal(), this.credentials.getToken()).createScanner(tableName, auths);
        return FindMax.findMax(scanner, startRow, startInclusive, endRow, endInclusive);
    }

    @Override
    public List<DiskUsage> getDiskUsage(Set<String> tableNames) throws AccumuloException, AccumuloSecurityException, TableNotFoundException {
        List<TDiskUsage> diskUsages = null;
        while (diskUsages == null) {
            Pair<String, ClientService.Client> pair = null;
            try {
                pair = ServerClient.getConnection(this.instance, false);
                diskUsages = pair.getSecond().getDiskUsage(tableNames, this.credentials.toThrift(this.instance));
            }
            catch (ThriftTableOperationException e) {
                switch (e.getType()) {
                    case NOTFOUND: {
                        throw new TableNotFoundException(e);
                    }
                    case NAMESPACE_NOTFOUND: {
                        throw new TableNotFoundException(e.getTableName(), new NamespaceNotFoundException(e));
                    }
                }
                throw new AccumuloException(e.description, (Throwable)((Object)e));
            }
            catch (ThriftSecurityException e) {
                throw new AccumuloSecurityException(e.getUser(), e.getCode());
            }
            catch (TTransportException e) {
                log.debug((Object)("disk usage request failed " + pair.getFirst() + ", retrying ... "), (Throwable)e);
                UtilWaitThread.sleep(100L);
            }
            catch (TException e) {
                throw new AccumuloException(e);
            }
            finally {
                if (pair == null) continue;
                ServerClient.close(pair.getSecond());
            }
        }
        ArrayList<DiskUsage> finalUsages = new ArrayList<DiskUsage>();
        for (TDiskUsage diskUsage : diskUsages) {
            finalUsages.add(new DiskUsage(new TreeSet<String>(diskUsage.getTables()), diskUsage.getUsage()));
        }
        return finalUsages;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static Map<String, String> getExportedProps(FileSystem fs, Path path) throws IOException {
        HashMap<String, String> props = new HashMap<String, String>();
        ZipInputStream zis = new ZipInputStream((InputStream)fs.open(path));
        try {
            ZipEntry zipEntry;
            while ((zipEntry = zis.getNextEntry()) != null) {
                if (!zipEntry.getName().equals("table_config.txt")) continue;
                BufferedReader in = new BufferedReader(new InputStreamReader((InputStream)zis, Constants.UTF8));
                try {
                    String line;
                    while ((line = in.readLine()) != null) {
                        String[] sa = line.split("=", 2);
                        props.put(sa[0], sa[1]);
                    }
                    return props;
                }
                finally {
                    in.close();
                    return props;
                }
            }
        }
        finally {
            zis.close();
        }
    }

    @Override
    public void importTable(String tableName, String importDir) throws TableExistsException, AccumuloException, AccumuloSecurityException {
        ArgumentChecker.notNull(tableName, importDir);
        try {
            importDir = this.checkPath(importDir, "Table", "").toString();
        }
        catch (IOException e) {
            throw new AccumuloException(e);
        }
        try {
            FileSystem fs = new Path(importDir).getFileSystem(CachedConfiguration.getInstance());
            Map<String, String> props = TableOperationsImpl.getExportedProps(fs, new Path(importDir, "exportMetadata.zip"));
            for (Map.Entry<String, String> entry : props.entrySet()) {
                if (!Property.isClassProperty(entry.getKey()) || entry.getValue().contains("org.apache.accumulo.core")) continue;
                Logger.getLogger(this.getClass()).info((Object)("Imported table sets '" + entry.getKey() + "' to '" + entry.getValue() + "'.  Ensure this class is on Accumulo classpath."));
            }
        }
        catch (IOException ioe) {
            Logger.getLogger(this.getClass()).warn((Object)("Failed to check if imported table references external java classes : " + ioe.getMessage()));
        }
        List<ByteBuffer> args = Arrays.asList(ByteBuffer.wrap(tableName.getBytes(Constants.UTF8)), ByteBuffer.wrap(importDir.getBytes(Constants.UTF8)));
        Map<String, String> opts = Collections.emptyMap();
        try {
            this.doTableFateOperation(tableName, AccumuloException.class, FateOperation.TABLE_IMPORT, args, opts);
        }
        catch (TableNotFoundException e) {
            throw new AssertionError((Object)e);
        }
    }

    @Override
    public void exportTable(String tableName, String exportDir) throws TableNotFoundException, AccumuloException, AccumuloSecurityException {
        ArgumentChecker.notNull(tableName, exportDir);
        List<ByteBuffer> args = Arrays.asList(ByteBuffer.wrap(tableName.getBytes(Constants.UTF8)), ByteBuffer.wrap(exportDir.getBytes(Constants.UTF8)));
        Map<String, String> opts = Collections.emptyMap();
        try {
            this.doTableFateOperation(tableName, TableNotFoundException.class, FateOperation.TABLE_EXPORT, args, opts);
        }
        catch (TableExistsException e) {
            throw new AssertionError((Object)e);
        }
    }

    @Override
    public boolean testClassLoad(final String tableName, final String className, final String asTypeName) throws TableNotFoundException, AccumuloException, AccumuloSecurityException {
        ArgumentChecker.notNull(tableName, className, asTypeName);
        try {
            return ServerClient.executeRaw(this.instance, new ClientExecReturn<Boolean, ClientService.Client>(){

                @Override
                public Boolean execute(ClientService.Client client) throws Exception {
                    return client.checkTableClass(Tracer.traceInfo(), TableOperationsImpl.this.credentials.toThrift(TableOperationsImpl.this.instance), tableName, className, asTypeName);
                }
            });
        }
        catch (ThriftTableOperationException e) {
            switch (e.getType()) {
                case NOTFOUND: {
                    throw new TableNotFoundException(e);
                }
                case NAMESPACE_NOTFOUND: {
                    throw new TableNotFoundException(tableName, new NamespaceNotFoundException(e));
                }
            }
            throw new AccumuloException(e.description, (Throwable)((Object)e));
        }
        catch (ThriftSecurityException e) {
            throw new AccumuloSecurityException(e.user, e.code, (Throwable)((Object)e));
        }
        catch (AccumuloException e) {
            throw e;
        }
        catch (Exception e) {
            throw new AccumuloException(e);
        }
    }

    @Override
    public void attachIterator(String tableName, IteratorSetting setting, EnumSet<IteratorUtil.IteratorScope> scopes) throws AccumuloSecurityException, AccumuloException, TableNotFoundException {
        this.testClassLoad(tableName, setting.getIteratorClass(), SortedKeyValueIterator.class.getName());
        super.attachIterator(tableName, setting, scopes);
    }

    @Override
    public int addConstraint(String tableName, String constraintClassName) throws AccumuloException, AccumuloSecurityException, TableNotFoundException {
        this.testClassLoad(tableName, constraintClassName, Constraint.class.getName());
        return super.addConstraint(tableName, constraintClassName);
    }

    private void doTableFateOperation(String tableName, Class<? extends Exception> namespaceNotFoundExceptionClass, FateOperation op, List<ByteBuffer> args, Map<String, String> opts) throws AccumuloSecurityException, AccumuloException, TableExistsException, TableNotFoundException {
        try {
            this.doFateOperation(op, args, opts);
        }
        catch (NamespaceExistsException e) {
            throw new AssertionError((Object)e);
        }
        catch (NamespaceNotFoundException e) {
            if (namespaceNotFoundExceptionClass == null) {
                throw new AssertionError((Object)e);
            }
            if (AccumuloException.class.isAssignableFrom(namespaceNotFoundExceptionClass)) {
                throw new AccumuloException("Cannot create table in non-existent namespace", e);
            }
            if (TableNotFoundException.class.isAssignableFrom(namespaceNotFoundExceptionClass)) {
                throw new TableNotFoundException(null, tableName, "Namespace not found", e);
            }
            throw new AssertionError((Object)e);
        }
    }

    private class SplitTask
    implements Runnable {
        private List<Text> splits;
        private SplitEnv env;

        SplitTask(SplitEnv env, List<Text> splits) {
            this.env = env;
            this.splits = splits;
        }

        @Override
        public void run() {
            try {
                if (this.env.exception.get() != null) {
                    return;
                }
                if (this.splits.size() <= 2) {
                    TableOperationsImpl.this.addSplits(this.env.tableName, new TreeSet<Text>(this.splits), this.env.tableId);
                    for (int i = 0; i < this.splits.size(); ++i) {
                        this.env.latch.countDown();
                    }
                    return;
                }
                int mid = this.splits.size() / 2;
                TableOperationsImpl.this.addSplits(this.env.tableName, new TreeSet<Text>(this.splits.subList(mid, mid + 1)), this.env.tableId);
                this.env.latch.countDown();
                this.env.executor.submit(new SplitTask(this.env, this.splits.subList(0, mid)));
                this.env.executor.submit(new SplitTask(this.env, this.splits.subList(mid + 1, this.splits.size())));
            }
            catch (Exception e) {
                this.env.exception.compareAndSet(null, e);
            }
        }
    }

    private static class SplitEnv {
        private String tableName;
        private String tableId;
        private ExecutorService executor;
        private CountDownLatch latch;
        private AtomicReference<Exception> exception;

        SplitEnv(String tableName, String tableId, ExecutorService executor, CountDownLatch latch, AtomicReference<Exception> exception) {
            this.tableName = tableName;
            this.tableId = tableId;
            this.executor = executor;
            this.latch = latch;
            this.exception = exception;
        }
    }
}

