/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.server.coordinator.duty;

import com.google.common.collect.Lists;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;
import org.apache.druid.client.ImmutableDruidServer;
import org.apache.druid.java.util.common.Pair;
import org.apache.druid.java.util.emitter.EmittingLogger;
import org.apache.druid.server.coordinator.BalancerSegmentHolder;
import org.apache.druid.server.coordinator.BalancerStrategy;
import org.apache.druid.server.coordinator.CoordinatorStats;
import org.apache.druid.server.coordinator.DruidCoordinator;
import org.apache.druid.server.coordinator.DruidCoordinatorRuntimeParams;
import org.apache.druid.server.coordinator.LoadPeonCallback;
import org.apache.druid.server.coordinator.LoadQueuePeon;
import org.apache.druid.server.coordinator.ServerHolder;
import org.apache.druid.server.coordinator.duty.CoordinatorDuty;
import org.apache.druid.timeline.DataSegment;
import org.apache.druid.timeline.SegmentId;

public class BalanceSegments
implements CoordinatorDuty {
    protected static final EmittingLogger log = new EmittingLogger(BalanceSegments.class);
    protected final DruidCoordinator coordinator;
    protected final Map<String, ConcurrentHashMap<SegmentId, BalancerSegmentHolder>> currentlyMovingSegments = new HashMap<String, ConcurrentHashMap<SegmentId, BalancerSegmentHolder>>();

    public BalanceSegments(DruidCoordinator coordinator) {
        this.coordinator = coordinator;
    }

    protected void reduceLifetimes(String tier) {
        for (BalancerSegmentHolder holder : this.currentlyMovingSegments.get(tier).values()) {
            holder.reduceLifetime();
            if (holder.getLifetime() > 0) continue;
            log.makeAlert("[%s]: Balancer move segments queue has a segment stuck", new Object[]{tier}).addData("segment", (Object)holder.getSegment().getId()).addData("server", (Object)holder.getFromServer().getMetadata()).emit();
        }
    }

    @Override
    public DruidCoordinatorRuntimeParams run(DruidCoordinatorRuntimeParams params) {
        CoordinatorStats stats = new CoordinatorStats();
        params.getDruidCluster().getHistoricals().forEach((tier, servers) -> this.balanceTier(params, (String)tier, (SortedSet<ServerHolder>)servers, stats));
        return params.buildFromExisting().withCoordinatorStats(stats).build();
    }

    private void balanceTier(DruidCoordinatorRuntimeParams params, String tier, SortedSet<ServerHolder> servers, CoordinatorStats stats) {
        log.info("Balancing segments in tier [%s]", new Object[]{tier});
        if (params.getUsedSegments().size() == 0) {
            log.info("Metadata segments are not available. Cannot balance.", new Object[0]);
            return;
        }
        this.currentlyMovingSegments.computeIfAbsent(tier, t -> new ConcurrentHashMap());
        if (!this.currentlyMovingSegments.get(tier).isEmpty()) {
            this.reduceLifetimes(tier);
            log.info("[%s]: Still waiting on %,d segments to be moved. Skipping balance.", new Object[]{tier, this.currentlyMovingSegments.get(tier).size()});
            return;
        }
        Map<Boolean, List<ServerHolder>> partitions = servers.stream().collect(Collectors.partitioningBy(ServerHolder::isDecommissioning));
        List<ServerHolder> decommissioningServers = partitions.get(true);
        List<ServerHolder> activeServers = partitions.get(false);
        log.info("Found %d active servers, %d decommissioning servers", new Object[]{activeServers.size(), decommissioningServers.size()});
        if (decommissioningServers.isEmpty() && activeServers.size() <= 1 || activeServers.isEmpty()) {
            log.warn("[%s]: insufficient active servers. Cannot balance.", new Object[]{tier});
            return;
        }
        int numSegments = 0;
        for (ServerHolder sourceHolder : servers) {
            numSegments += sourceHolder.getServer().getNumSegments();
        }
        if (numSegments == 0) {
            log.info("No segments found. Cannot balance.", new Object[0]);
            return;
        }
        int maxSegmentsToMove = Math.min(params.getCoordinatorDynamicConfig().getMaxSegmentsToMove(), numSegments);
        int decommissioningMaxPercentOfMaxSegmentsToMove = params.getCoordinatorDynamicConfig().getDecommissioningMaxPercentOfMaxSegmentsToMove();
        int maxSegmentsToMoveFromDecommissioningNodes = (int)Math.ceil((double)maxSegmentsToMove * ((double)decommissioningMaxPercentOfMaxSegmentsToMove / 100.0));
        log.info("Processing %d segments for moving from decommissioning servers", new Object[]{maxSegmentsToMoveFromDecommissioningNodes});
        Pair<Integer, Integer> decommissioningResult = this.balanceServers(params, decommissioningServers, activeServers, maxSegmentsToMoveFromDecommissioningNodes);
        int maxGeneralSegmentsToMove = maxSegmentsToMove - (Integer)decommissioningResult.lhs;
        log.info("Processing %d segments for balancing between active servers", new Object[]{maxGeneralSegmentsToMove});
        Pair<Integer, Integer> generalResult = this.balanceServers(params, activeServers, activeServers, maxGeneralSegmentsToMove);
        int moved = (Integer)generalResult.lhs + (Integer)decommissioningResult.lhs;
        int unmoved = (Integer)generalResult.rhs + (Integer)decommissioningResult.rhs;
        if (unmoved == maxSegmentsToMove) {
            log.info("No good moves found in tier [%s]", new Object[]{tier});
        }
        stats.addToTieredStat("unmovedCount", tier, unmoved);
        stats.addToTieredStat("movedCount", tier, moved);
        if (params.getCoordinatorDynamicConfig().emitBalancingStats()) {
            BalancerStrategy strategy = params.getBalancerStrategy();
            strategy.emitStats(tier, stats, Lists.newArrayList(servers));
        }
        log.info("[%s]: Segments Moved: [%d] Segments Let Alone: [%d]", new Object[]{tier, moved, unmoved});
    }

    private Pair<Integer, Integer> balanceServers(DruidCoordinatorRuntimeParams params, List<ServerHolder> toMoveFrom, List<ServerHolder> toMoveTo, int maxSegmentsToMove) {
        if (maxSegmentsToMove <= 0) {
            log.debug("maxSegmentsToMove is 0; no balancing work can be performed.", new Object[0]);
            return new Pair((Object)0, (Object)0);
        }
        if (toMoveFrom.isEmpty()) {
            log.debug("toMoveFrom is empty; no balancing work can be performed.", new Object[0]);
            return new Pair((Object)0, (Object)0);
        }
        if (toMoveTo.isEmpty()) {
            log.debug("toMoveTo is empty; no balancing work can be peformed.", new Object[0]);
            return new Pair((Object)0, (Object)0);
        }
        BalancerStrategy strategy = params.getBalancerStrategy();
        int maxIterations = 2 * maxSegmentsToMove;
        int maxToLoad = params.getCoordinatorDynamicConfig().getMaxSegmentsInNodeLoadingQueue();
        int moved = 0;
        int unmoved = 0;
        Iterator<BalancerSegmentHolder> segmentsToMove = params.getCoordinatorDynamicConfig().useBatchedSegmentSampler() ? strategy.pickSegmentsToMove(toMoveFrom, params.getBroadcastDatasources(), maxSegmentsToMove) : strategy.pickSegmentsToMove(toMoveFrom, params.getBroadcastDatasources(), params.getCoordinatorDynamicConfig().getPercentOfSegmentsToConsiderPerMove());
        int iter = 0;
        while (moved + unmoved < maxSegmentsToMove) {
            if (!segmentsToMove.hasNext()) {
                log.info("All servers to move segments from are empty, ending run.", new Object[0]);
                break;
            }
            BalancerSegmentHolder segmentToMoveHolder = segmentsToMove.next();
            boolean needToBalancePickedSegment = params.getUsedSegments().contains(segmentToMoveHolder.getSegment());
            if (needToBalancePickedSegment) {
                DataSegment segmentToMove = segmentToMoveHolder.getSegment();
                ImmutableDruidServer fromServer = segmentToMoveHolder.getFromServer();
                List<ServerHolder> toMoveToWithLoadQueueCapacityAndNotServingSegment = toMoveTo.stream().filter(s -> s.getServer().equals(fromServer) || !s.isServingSegment(segmentToMove) && (maxToLoad <= 0 || s.getNumberOfSegmentsInQueue() < maxToLoad)).collect(Collectors.toList());
                if (toMoveToWithLoadQueueCapacityAndNotServingSegment.size() > 0) {
                    ServerHolder destinationHolder = strategy.findNewSegmentHomeBalancer(segmentToMove, toMoveToWithLoadQueueCapacityAndNotServingSegment);
                    if (destinationHolder != null && !destinationHolder.getServer().equals(fromServer)) {
                        if (this.moveSegment(segmentToMoveHolder, destinationHolder.getServer(), params)) {
                            ++moved;
                        } else {
                            ++unmoved;
                        }
                    } else {
                        log.debug("Segment [%s] is 'optimally' placed.", new Object[]{segmentToMove.getId()});
                        ++unmoved;
                    }
                } else {
                    log.debug("No valid movement destinations for segment [%s].", new Object[]{segmentToMove.getId()});
                    ++unmoved;
                }
            }
            if (iter >= maxIterations) {
                log.info("Unable to select %d remaining candidate segments out of %d total to balance after %d iterations, ending run.", new Object[]{maxSegmentsToMove - moved - unmoved, maxSegmentsToMove, iter});
                break;
            }
            ++iter;
        }
        return new Pair((Object)moved, (Object)unmoved);
    }

    protected boolean moveSegment(BalancerSegmentHolder segment, ImmutableDruidServer toServer, DruidCoordinatorRuntimeParams params) {
        LoadQueuePeon toPeon = params.getLoadManagementPeons().get(toServer.getName());
        ImmutableDruidServer fromServer = segment.getFromServer();
        DataSegment segmentToMove = segment.getSegment();
        SegmentId segmentId = segmentToMove.getId();
        if (!toPeon.getSegmentsToLoad().contains(segmentToMove) && toServer.getSegment(segmentId) == null && new ServerHolder(toServer, toPeon).getAvailableSize() > segmentToMove.getSize()) {
            log.debug("Moving [%s] from [%s] to [%s]", new Object[]{segmentId, fromServer.getName(), toServer.getName()});
            ConcurrentMap movingSegments = this.currentlyMovingSegments.get(toServer.getTier());
            movingSegments.put(segmentId, segment);
            LoadPeonCallback callback = moveSuccess -> {
                BalancerSegmentHolder cfr_ignored_0 = (BalancerSegmentHolder)movingSegments.remove(segmentId);
            };
            try {
                this.coordinator.moveSegment(params, fromServer, toServer, segmentToMove, callback);
                return true;
            }
            catch (Exception e) {
                log.makeAlert((Throwable)e, "[%s] : Moving exception", new Object[]{segmentId}).emit();
                callback.execute(false);
            }
        }
        return false;
    }
}

