/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.java.util.common.io.smoosh;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.primitives.Ints;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.GatheringByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.druid.java.util.common.FileUtils;
import org.apache.druid.java.util.common.IAE;
import org.apache.druid.java.util.common.IOE;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.MappedByteBufferHandler;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.java.util.common.io.Closer;
import org.apache.druid.java.util.common.io.smoosh.Metadata;
import org.apache.druid.java.util.common.io.smoosh.SmooshedWriter;
import org.apache.druid.java.util.common.logger.Logger;

public class FileSmoosher
implements Closeable {
    private static final String FILE_EXTENSION = "smoosh";
    private static final Joiner JOINER = Joiner.on((String)",");
    private static final Logger LOG = new Logger(FileSmoosher.class);
    private final File baseDir;
    private final int maxChunkSize;
    private final List<File> outFiles = new ArrayList<File>();
    private final Map<String, Metadata> internalFiles = new TreeMap<String, Metadata>();
    private List<File> completedFiles = new ArrayList<File>();
    private List<File> filesInProcess = new ArrayList<File>();
    private AtomicLong delegateFileCounter = new AtomicLong(0L);
    private Map<String, String> delegateFileNameMap;
    private Outer currOut = null;
    private boolean writerCurrentlyInUse = false;

    public FileSmoosher(File baseDir) {
        this(baseDir, Integer.MAX_VALUE);
    }

    public FileSmoosher(File baseDir, int maxChunkSize) {
        this.baseDir = baseDir;
        this.maxChunkSize = maxChunkSize;
        this.delegateFileNameMap = new HashMap<String, String>();
        Preconditions.checkArgument((maxChunkSize > 0 ? 1 : 0) != 0, (Object)"maxChunkSize must be a positive value.");
    }

    static File metaFile(File baseDir) {
        return new File(baseDir, StringUtils.format("meta.%s", FILE_EXTENSION));
    }

    static File makeChunkFile(File baseDir, int i) {
        return new File(baseDir, StringUtils.format("%05d.%s", i, FILE_EXTENSION));
    }

    public void add(File fileToAdd) throws IOException {
        this.add(fileToAdd.getName(), fileToAdd);
    }

    public void add(String name, File fileToAdd) throws IOException {
        try (MappedByteBufferHandler fileMappingHandler = FileUtils.map(fileToAdd);){
            this.add(name, fileMappingHandler.get());
        }
    }

    public void add(String name, ByteBuffer bufferToAdd) throws IOException {
        this.add(name, Collections.singletonList(bufferToAdd));
    }

    public void add(String name, List<ByteBuffer> bufferToAdd) throws IOException {
        if (name.contains(",")) {
            throw new IAE("Cannot have a comma in the name of a file, got[%s].", name);
        }
        if (this.internalFiles.get(name) != null) {
            throw new IAE("Cannot add files of the same name, already have [%s]", name);
        }
        long size = 0L;
        for (ByteBuffer buffer : bufferToAdd) {
            size += (long)buffer.remaining();
        }
        try (SmooshedWriter out = this.addWithSmooshedWriter(name, size);){
            for (ByteBuffer buffer : bufferToAdd) {
                out.write(buffer);
            }
        }
    }

    public SmooshedWriter addWithSmooshedWriter(final String name, final long size) throws IOException {
        if (size > (long)this.maxChunkSize) {
            throw new IAE("Asked to add buffers[%,d] larger than configured max[%,d]", size, this.maxChunkSize);
        }
        if (this.writerCurrentlyInUse) {
            return this.delegateSmooshedWriter(name, size);
        }
        if (this.currOut == null) {
            this.currOut = this.getNewCurrOut();
        }
        if ((long)this.currOut.bytesLeft() < size) {
            this.currOut.close();
            this.currOut = this.getNewCurrOut();
        }
        final int startOffset = this.currOut.getCurrOffset();
        this.writerCurrentlyInUse = true;
        return new SmooshedWriter(){
            private boolean open = true;
            private long bytesWritten = 0L;

            @Override
            public int write(ByteBuffer in) throws IOException {
                return this.verifySize(FileSmoosher.this.currOut.write(in));
            }

            @Override
            public long write(ByteBuffer[] srcs, int offset, int length) throws IOException {
                return this.verifySize(FileSmoosher.this.currOut.write(srcs, offset, length));
            }

            @Override
            public long write(ByteBuffer[] srcs) throws IOException {
                return this.verifySize(FileSmoosher.this.currOut.write(srcs));
            }

            private int verifySize(long bytesWrittenInChunk) {
                this.bytesWritten += bytesWrittenInChunk;
                if (this.bytesWritten != (long)(FileSmoosher.this.currOut.getCurrOffset() - startOffset)) {
                    throw new ISE("Perhaps there is some concurrent modification going on?", new Object[0]);
                }
                if (this.bytesWritten > size) {
                    throw new ISE("Wrote[%,d] bytes for something of size[%,d].  Liar!!!", this.bytesWritten, size);
                }
                return Ints.checkedCast((long)bytesWrittenInChunk);
            }

            @Override
            public boolean isOpen() {
                return this.open;
            }

            @Override
            public void close() throws IOException {
                if (this.open) {
                    this.open = false;
                    FileSmoosher.this.internalFiles.put(name, new Metadata(FileSmoosher.this.currOut.getFileNum(), startOffset, FileSmoosher.this.currOut.getCurrOffset()));
                    FileSmoosher.this.writerCurrentlyInUse = false;
                    if (this.bytesWritten != (long)(FileSmoosher.this.currOut.getCurrOffset() - startOffset)) {
                        throw new ISE("Perhaps there is some concurrent modification going on?", new Object[0]);
                    }
                    if (this.bytesWritten != size) {
                        throw new IOE("Expected [%,d] bytes, only saw [%,d], potential corruption?", size, this.bytesWritten);
                    }
                    FileSmoosher.this.mergeWithSmoosher();
                }
            }
        };
    }

    private void mergeWithSmoosher() throws IOException {
        ArrayList<File> fileToProcess = new ArrayList<File>(this.completedFiles);
        ImmutableMap fileNameMap = ImmutableMap.copyOf(this.delegateFileNameMap);
        this.completedFiles = new ArrayList<File>();
        this.delegateFileNameMap = new HashMap<String, String>();
        for (File file : fileToProcess) {
            this.add((String)fileNameMap.get(file.getName()), file);
            if (file.delete()) continue;
            LOG.warn("Unable to delete file [%s]", file);
        }
    }

    private SmooshedWriter delegateSmooshedWriter(String name, final long size) throws IOException {
        String delegateName = this.getDelegateFileName(name);
        final File tmpFile = new File(this.baseDir, delegateName);
        this.filesInProcess.add(tmpFile);
        return new SmooshedWriter(){
            private final GatheringByteChannel channel;
            private int currOffset;
            {
                this.channel = FileChannel.open(tmpFile.toPath(), StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
                this.currOffset = 0;
            }

            @Override
            public void close() throws IOException {
                this.channel.close();
                FileSmoosher.this.completedFiles.add(tmpFile);
                FileSmoosher.this.filesInProcess.remove(tmpFile);
                if (!FileSmoosher.this.writerCurrentlyInUse) {
                    FileSmoosher.this.mergeWithSmoosher();
                }
            }

            public int bytesLeft() {
                return (int)(size - (long)this.currOffset);
            }

            @Override
            public int write(ByteBuffer buffer) throws IOException {
                return this.addToOffset(this.channel.write(buffer));
            }

            @Override
            public long write(ByteBuffer[] srcs, int offset, int length) throws IOException {
                return this.addToOffset(this.channel.write(srcs, offset, length));
            }

            @Override
            public long write(ByteBuffer[] srcs) throws IOException {
                return this.addToOffset(this.channel.write(srcs));
            }

            public int addToOffset(long numBytesWritten) {
                if (numBytesWritten > (long)this.bytesLeft()) {
                    throw new ISE("Wrote more bytes[%,d] than available[%,d]. Don't do that.", numBytesWritten, this.bytesLeft());
                }
                this.currOffset = (int)((long)this.currOffset + numBytesWritten);
                return Ints.checkedCast((long)numBytesWritten);
            }

            @Override
            public boolean isOpen() {
                return this.channel.isOpen();
            }
        };
    }

    private String getDelegateFileName(String name) {
        String delegateName = String.valueOf(this.delegateFileCounter.getAndIncrement());
        this.delegateFileNameMap.put(delegateName, name);
        return delegateName;
    }

    @Override
    public void close() throws IOException {
        if (!this.completedFiles.isEmpty() || !this.filesInProcess.isEmpty()) {
            for (File file : this.completedFiles) {
                if (file.delete()) continue;
                LOG.warn("Unable to delete file [%s]", file);
            }
            for (File file : this.filesInProcess) {
                if (file.delete()) continue;
                LOG.warn("Unable to delete file [%s]", file);
            }
            throw new ISE("[%d] writers in progress and [%d] completed writers needs to be closed before closing smoosher.", this.filesInProcess.size(), this.completedFiles.size());
        }
        if (this.currOut != null) {
            this.currOut.close();
        }
        File metaFile = FileSmoosher.metaFile(this.baseDir);
        try (BufferedWriter out = new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(metaFile), StandardCharsets.UTF_8));){
            out.write(StringUtils.format("v1,%d,%d", this.maxChunkSize, this.outFiles.size()));
            out.write("\n");
            for (Map.Entry<String, Metadata> entry : this.internalFiles.entrySet()) {
                Metadata metadata = entry.getValue();
                out.write(JOINER.join((Object)entry.getKey(), (Object)metadata.getFileNum(), new Object[]{metadata.getStartOffset(), metadata.getEndOffset()}));
                out.write("\n");
            }
        }
    }

    private Outer getNewCurrOut() throws FileNotFoundException {
        int fileNum = this.outFiles.size();
        File outFile = FileSmoosher.makeChunkFile(this.baseDir, fileNum);
        this.outFiles.add(outFile);
        return new Outer(fileNum, outFile, this.maxChunkSize);
    }

    public static class Outer
    implements SmooshedWriter {
        private final int fileNum;
        private final int maxLength;
        private final File outFile;
        private final GatheringByteChannel channel;
        private final Closer closer = Closer.create();
        private int currOffset = 0;

        Outer(int fileNum, File outFile, int maxLength) throws FileNotFoundException {
            this.fileNum = fileNum;
            this.outFile = outFile;
            this.maxLength = maxLength;
            FileOutputStream outStream = this.closer.register(new FileOutputStream(outFile));
            this.channel = this.closer.register(outStream.getChannel());
        }

        public int getFileNum() {
            return this.fileNum;
        }

        public int getCurrOffset() {
            return this.currOffset;
        }

        public int bytesLeft() {
            return this.maxLength - this.currOffset;
        }

        @Override
        public int write(ByteBuffer buffer) throws IOException {
            return this.addToOffset(this.channel.write(buffer));
        }

        @Override
        public long write(ByteBuffer[] srcs, int offset, int length) throws IOException {
            return this.addToOffset(this.channel.write(srcs, offset, length));
        }

        @Override
        public long write(ByteBuffer[] srcs) throws IOException {
            return this.addToOffset(this.channel.write(srcs));
        }

        public int addToOffset(long numBytesWritten) {
            if (numBytesWritten > (long)this.bytesLeft()) {
                throw new ISE("Wrote more bytes[%,d] than available[%,d]. Don't do that.", numBytesWritten, this.bytesLeft());
            }
            this.currOffset = (int)((long)this.currOffset + numBytesWritten);
            return Ints.checkedCast((long)numBytesWritten);
        }

        @Override
        public boolean isOpen() {
            return this.channel.isOpen();
        }

        @Override
        public void close() throws IOException {
            this.closer.close();
            LOG.debug("Created smoosh file [%s] of size [%s] bytes.", this.outFile.getAbsolutePath(), this.outFile.length());
        }
    }
}

