// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.rules;

import org.apache.doris.nereids.pattern.PatternMatcher;
import org.apache.doris.nereids.trees.plans.Plan;

/**
 * Type of rules, each rule has its unique type.
 */
public enum RuleType {
    // just for UT
    TEST_REWRITE(RuleTypeClass.REWRITE),
    // binding rules

    // **** make sure BINDING_UNBOUND_LOGICAL_PLAN is the lowest priority in the rewrite rules. ****
    BINDING_NON_LEAF_LOGICAL_PLAN(RuleTypeClass.REWRITE),
    BINDING_ONE_ROW_RELATION_SLOT(RuleTypeClass.REWRITE),
    BINDING_RELATION(RuleTypeClass.REWRITE),
    BINDING_PROJECT_SLOT(RuleTypeClass.REWRITE),
    BINDING_JOIN_SLOT(RuleTypeClass.REWRITE),
    BINDING_FILTER_SLOT(RuleTypeClass.REWRITE),
    BINDING_AGGREGATE_SLOT(RuleTypeClass.REWRITE),
    BINDING_REPEAT_SLOT(RuleTypeClass.REWRITE),
    BINDING_HAVING_SLOT(RuleTypeClass.REWRITE),
    BINDING_SORT_SLOT(RuleTypeClass.REWRITE),
    BINDING_LIMIT_SLOT(RuleTypeClass.REWRITE),
    BINDING_ONE_ROW_RELATION_FUNCTION(RuleTypeClass.REWRITE),
    BINDING_PROJECT_FUNCTION(RuleTypeClass.REWRITE),
    BINDING_AGGREGATE_FUNCTION(RuleTypeClass.REWRITE),
    BINDING_REPEAT_FUNCTION(RuleTypeClass.REWRITE),
    BINDING_SUBQUERY_ALIAS_SLOT(RuleTypeClass.REWRITE),
    BINDING_FILTER_FUNCTION(RuleTypeClass.REWRITE),
    BINDING_HAVING_FUNCTION(RuleTypeClass.REWRITE),
    BINDING_SORT_FUNCTION(RuleTypeClass.REWRITE),
    BINDING_UNBOUND_TVF_RELATION_FUNCTION(RuleTypeClass.REWRITE),

    REPLACE_SORT_EXPRESSION_BY_CHILD_OUTPUT(RuleTypeClass.REWRITE),

    FILL_UP_HAVING_AGGREGATE(RuleTypeClass.REWRITE),
    FILL_UP_SORT_AGGREGATE(RuleTypeClass.REWRITE),
    FILL_UP_SORT_HAVING_AGGREGATE(RuleTypeClass.REWRITE),
    FILL_UP_SORT_PROJECT(RuleTypeClass.REWRITE),

    RESOLVE_PROJECT_ALIAS(RuleTypeClass.REWRITE),
    RESOLVE_AGGREGATE_ALIAS(RuleTypeClass.REWRITE),
    PROJECT_TO_GLOBAL_AGGREGATE(RuleTypeClass.REWRITE),
    REGISTER_CTE(RuleTypeClass.REWRITE),

    RELATION_AUTHENTICATION(RuleTypeClass.VALIDATION),

    ADJUST_NULLABLE_FOR_PROJECT_SLOT(RuleTypeClass.REWRITE),
    ADJUST_NULLABLE_FOR_AGGREGATE_SLOT(RuleTypeClass.REWRITE),
    ADJUST_NULLABLE_FOR_REPEAT_SLOT(RuleTypeClass.REWRITE),

    CHECK_ROW_POLICY(RuleTypeClass.REWRITE),

    // check analysis rule
    CHECK_ANALYSIS(RuleTypeClass.CHECK),

    // rewrite rules
    NORMALIZE_AGGREGATE(RuleTypeClass.REWRITE),
    NORMALIZE_REPEAT(RuleTypeClass.REWRITE),
    AGGREGATE_DISASSEMBLE(RuleTypeClass.REWRITE),
    COLUMN_PRUNE_PROJECTION(RuleTypeClass.REWRITE),
    ELIMINATE_UNNECESSARY_PROJECT(RuleTypeClass.REWRITE),
    LOGICAL_SUB_QUERY_ALIAS_TO_LOGICAL_PROJECT(RuleTypeClass.REWRITE),
    ELIMINATE_GROUP_BY_CONSTANT(RuleTypeClass.REWRITE),

    // subquery analyze
    ANALYZE_FILTER_SUBQUERY(RuleTypeClass.REWRITE),
    // subquery rewrite rule
    PUSH_APPLY_UNDER_PROJECT(RuleTypeClass.REWRITE),
    PUSH_APPLY_UNDER_FILTER(RuleTypeClass.REWRITE),
    APPLY_PULL_FILTER_ON_AGG(RuleTypeClass.REWRITE),
    APPLY_PULL_FILTER_ON_PROJECT_UNDER_AGG(RuleTypeClass.REWRITE),
    SCALAR_APPLY_TO_JOIN(RuleTypeClass.REWRITE),
    IN_APPLY_TO_JOIN(RuleTypeClass.REWRITE),
    EXISTS_APPLY_TO_JOIN(RuleTypeClass.REWRITE),
    // predicate push down rules
    PUSHDOWN_JOIN_OTHER_CONDITION(RuleTypeClass.REWRITE),
    PUSHDOWN_PREDICATE_THROUGH_AGGREGATION(RuleTypeClass.REWRITE),
    PUSHDOWN_PREDICATE_THROUGH_REPEAT(RuleTypeClass.REWRITE),
    PUSHDOWN_EXPRESSIONS_IN_HASH_CONDITIONS(RuleTypeClass.REWRITE),
    // Pushdown filter
    PUSHDOWN_FILTER_THROUGH_JOIN(RuleTypeClass.REWRITE),
    PUSHDOWN_FILTER_THROUGH_LEFT_SEMI_JOIN(RuleTypeClass.REWRITE),
    PUSH_FILTER_INSIDE_JOIN(RuleTypeClass.REWRITE),
    PUSHDOWN_FILTER_THROUGH_PROJECT(RuleTypeClass.REWRITE),
    PUSHDOWN_PROJECT_THROUGHT_LIMIT(RuleTypeClass.REWRITE),
    // column prune rules,
    COLUMN_PRUNE_AGGREGATION_CHILD(RuleTypeClass.REWRITE),
    COLUMN_PRUNE_FILTER_CHILD(RuleTypeClass.REWRITE),
    COLUMN_PRUNE_SORT_CHILD(RuleTypeClass.REWRITE),
    COLUMN_PRUNE_JOIN_CHILD(RuleTypeClass.REWRITE),
    COLUMN_PRUNE_REPEAT_CHILD(RuleTypeClass.REWRITE),
    // expression of plan rewrite
    REWRITE_ONE_ROW_RELATION_EXPRESSION(RuleTypeClass.REWRITE),
    REWRITE_PROJECT_EXPRESSION(RuleTypeClass.REWRITE),
    REWRITE_AGG_EXPRESSION(RuleTypeClass.REWRITE),
    REWRITE_FILTER_EXPRESSION(RuleTypeClass.REWRITE),
    REWRITE_JOIN_EXPRESSION(RuleTypeClass.REWRITE),
    REORDER_JOIN(RuleTypeClass.REWRITE),
    // Merge Consecutive plan
    MERGE_FILTERS(RuleTypeClass.REWRITE),
    MERGE_PROJECTS(RuleTypeClass.REWRITE),
    MERGE_LIMITS(RuleTypeClass.REWRITE),
    // Eliminate plan
    ELIMINATE_LIMIT(RuleTypeClass.REWRITE),
    ELIMINATE_FILTER(RuleTypeClass.REWRITE),
    ELIMINATE_OUTER_JOIN(RuleTypeClass.REWRITE),
    FIND_HASH_CONDITION_FOR_JOIN(RuleTypeClass.REWRITE),
    MATERIALIZED_INDEX_AGG_SCAN(RuleTypeClass.REWRITE),
    MATERIALIZED_INDEX_AGG_FILTER_SCAN(RuleTypeClass.REWRITE),
    MATERIALIZED_INDEX_AGG_PROJECT_SCAN(RuleTypeClass.REWRITE),
    MATERIALIZED_INDEX_AGG_PROJECT_FILTER_SCAN(RuleTypeClass.REWRITE),
    MATERIALIZED_INDEX_AGG_FILTER_PROJECT_SCAN(RuleTypeClass.REWRITE),
    MATERIALIZED_INDEX_SCAN(RuleTypeClass.REWRITE),
    MATERIALIZED_INDEX_FILTER_SCAN(RuleTypeClass.REWRITE),
    MATERIALIZED_INDEX_PROJECT_SCAN(RuleTypeClass.REWRITE),
    MATERIALIZED_INDEX_PROJECT_FILTER_SCAN(RuleTypeClass.REWRITE),
    MATERIALIZED_INDEX_FILTER_PROJECT_SCAN(RuleTypeClass.REWRITE),
    OLAP_SCAN_PARTITION_PRUNE(RuleTypeClass.REWRITE),
    OLAP_SCAN_TABLET_PRUNE(RuleTypeClass.REWRITE),
    EXTRACT_SINGLE_TABLE_EXPRESSION_FROM_DISJUNCTION(RuleTypeClass.REWRITE),
    REWRITE_SENTINEL(RuleTypeClass.REWRITE),

    // limit push down
    PUSH_LIMIT_THROUGH_JOIN(RuleTypeClass.REWRITE),
    PUSH_LIMIT_THROUGH_PROJECT_JOIN(RuleTypeClass.REWRITE),

    // Join Reorder
    JOIN_REORDER(RuleTypeClass.REWRITE),

    // exploration rules
    TEST_EXPLORATION(RuleTypeClass.EXPLORATION),
    LOGICAL_JOIN_COMMUTATE(RuleTypeClass.EXPLORATION),
    LOGICAL_INNER_JOIN_LASSCOM(RuleTypeClass.EXPLORATION),
    LOGICAL_INNER_JOIN_LASSCOM_PROJECT(RuleTypeClass.EXPLORATION),
    LOGICAL_OUTER_JOIN_LASSCOM(RuleTypeClass.EXPLORATION),
    LOGICAL_OUTER_JOIN_LASSCOM_PROJECT(RuleTypeClass.EXPLORATION),
    LOGICAL_SEMI_JOIN_LOGICAL_JOIN_TRANSPOSE(RuleTypeClass.EXPLORATION),
    LOGICAL_SEMI_JOIN_LOGICAL_JOIN_TRANSPOSE_PROJECT(RuleTypeClass.EXPLORATION),
    LOGICAL_SEMI_JOIN_SEMI_JOIN_TRANPOSE(RuleTypeClass.EXPLORATION),
    LOGICAL_JOIN_EXCHANGE(RuleTypeClass.EXPLORATION),
    LOGICAL_JOIN_EXCHANGE_LEFT_PROJECT(RuleTypeClass.EXPLORATION),
    LOGICAL_JOIN_EXCHANGE_RIGHT_PROJECT(RuleTypeClass.EXPLORATION),
    LOGICAL_JOIN_EXCHANGE_BOTH_PROJECT(RuleTypeClass.EXPLORATION),
    LOGICAL_INNER_JOIN_LEFT_ASSOCIATIVE(RuleTypeClass.EXPLORATION),
    LOGICAL_INNER_JOIN_LEFT_ASSOCIATIVE_PROJECT(RuleTypeClass.EXPLORATION),
    LOGICAL_INNER_JOIN_RIGHT_ASSOCIATIVE(RuleTypeClass.EXPLORATION),
    LOGICAL_INNER_JOIN_RIGHT_ASSOCIATIVE_PROJECT(RuleTypeClass.EXPLORATION),

    // implementation rules
    LOGICAL_ONE_ROW_RELATION_TO_PHYSICAL_ONE_ROW_RELATION(RuleTypeClass.IMPLEMENTATION),
    LOGICAL_TVF_RELATION_TO_PHYSICAL_TVF_RELATION(RuleTypeClass.IMPLEMENTATION),
    LOGICAL_AGG_TO_PHYSICAL_HASH_AGG_RULE(RuleTypeClass.IMPLEMENTATION),
    LOGICAL_JOIN_TO_HASH_JOIN_RULE(RuleTypeClass.IMPLEMENTATION),
    LOGICAL_JOIN_TO_NESTED_LOOP_JOIN_RULE(RuleTypeClass.IMPLEMENTATION),
    LOGICAL_PROJECT_TO_PHYSICAL_PROJECT_RULE(RuleTypeClass.IMPLEMENTATION),
    LOGICAL_FILTER_TO_PHYSICAL_FILTER_RULE(RuleTypeClass.IMPLEMENTATION),
    LOGICAL_SORT_TO_PHYSICAL_QUICK_SORT_RULE(RuleTypeClass.IMPLEMENTATION),
    LOGICAL_TOP_N_TO_PHYSICAL_TOP_N_RULE(RuleTypeClass.IMPLEMENTATION),
    LOGICAL_EMPTY_RELATION_TO_PHYSICAL_EMPTY_RELATION_RULE(RuleTypeClass.IMPLEMENTATION),
    LOGICAL_LIMIT_TO_PHYSICAL_LIMIT_RULE(RuleTypeClass.IMPLEMENTATION),
    LOGICAL_OLAP_SCAN_TO_PHYSICAL_OLAP_SCAN_RULE(RuleTypeClass.IMPLEMENTATION),
    LOGICAL_ASSERT_NUM_ROWS_TO_PHYSICAL_ASSERT_NUM_ROWS(RuleTypeClass.IMPLEMENTATION),
    IMPLEMENTATION_SENTINEL(RuleTypeClass.IMPLEMENTATION),

    LOGICAL_SEMI_JOIN_SEMI_JOIN_TRANPOSE_PROJECT(RuleTypeClass.EXPLORATION),
    // sentinel, use to count rules
    SENTINEL(RuleTypeClass.SENTINEL),
    ;

    private final RuleTypeClass ruleTypeClass;

    RuleType(RuleTypeClass ruleTypeClass) {
        this.ruleTypeClass = ruleTypeClass;
    }

    public int type() {
        return ordinal();
    }

    public RuleTypeClass getRuleTypeClass() {
        return ruleTypeClass;
    }

    public <INPUT_TYPE extends Plan, OUTPUT_TYPE extends Plan> Rule build(
            PatternMatcher<INPUT_TYPE, OUTPUT_TYPE> patternMatcher) {
        return patternMatcher.toRule(this);
    }

    enum RuleTypeClass {
        REWRITE,
        EXPLORATION,
        // This type is used for unit test only.
        CHECK,
        IMPLEMENTATION,
        VALIDATION,
        SENTINEL,
        ;
    }
}
